<?php
/**
 * Advanced Admin Panel - Bootstrap 5 Version
 * Modern, responsive, and feature-rich admin interface
 */

// Enable error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Set timezone
date_default_timezone_set('Asia/Jakarta');

// Load environment variables
if (file_exists('.env')) {
    $lines = file('.env', FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    foreach ($lines as $line) {
        if (strpos($line, '=') !== false && strpos($line, '#') !== 0) {
            list($key, $value) = explode('=', $line, 2);
            $_ENV[trim($key)] = trim($value);
        }
    }
}

// Load dependencies
require_once 'config/database.php';
require_once 'classes/DatabaseLogger.php';

// Authentication check
session_start();
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    if (isset($_POST['username']) && isset($_POST['password'])) {
        $username = $_POST['username'];
        $password = $_POST['password'];
        
        // Simple authentication (you can enhance this)
        if ($username === 'admin' && $password === 'admin123') {
            $_SESSION['admin_logged_in'] = true;
            $_SESSION['admin_username'] = $username;
        } else {
            $login_error = 'Invalid credentials';
        }
    }
    
    if (!isset($_SESSION['admin_logged_in'])) {
        // Show login form
        ?>
        <!DOCTYPE html>
        <html lang="en">
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <title>Admin Login - Telegram Bot</title>
            <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
            <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
            <style>
                body { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); min-height: 100vh; }
                .login-card { backdrop-filter: blur(10px); background: rgba(255,255,255,0.1); border: 1px solid rgba(255,255,255,0.2); }
                .glass-effect { background: rgba(255,255,255,0.1); backdrop-filter: blur(10px); border: 1px solid rgba(255,255,255,0.2); }
            </style>
        </head>
        <body class="d-flex align-items-center justify-content-center">
            <div class="container">
                <div class="row justify-content-center">
                    <div class="col-md-6 col-lg-4">
                        <div class="card login-card shadow-lg border-0">
                            <div class="card-body p-5">
                                <div class="text-center mb-4">
                                    <i class="fas fa-robot fa-3x text-white mb-3"></i>
                                    <h3 class="text-white">Admin Login</h3>
                                    <p class="text-white-50">Telegram Bot Management</p>
                                </div>
                                
                                <?php if (isset($login_error)): ?>
                                    <div class="alert alert-danger" role="alert">
                                        <i class="fas fa-exclamation-triangle me-2"></i><?= $login_error ?>
                                    </div>
                                <?php endif; ?>
                                
                                <form method="POST">
                                    <div class="mb-3">
                                        <label class="form-label text-white">Username</label>
                                        <div class="input-group">
                                            <span class="input-group-text glass-effect border-0"><i class="fas fa-user text-white"></i></span>
                                            <input type="text" class="form-control glass-effect border-0 text-white" name="username" placeholder="Enter username" required>
                                        </div>
                                    </div>
                                    <div class="mb-4">
                                        <label class="form-label text-white">Password</label>
                                        <div class="input-group">
                                            <span class="input-group-text glass-effect border-0"><i class="fas fa-lock text-white"></i></span>
                                            <input type="password" class="form-control glass-effect border-0 text-white" name="password" placeholder="Enter password" required>
                                        </div>
                                    </div>
                                    <button type="submit" class="btn btn-light w-100 py-2">
                                        <i class="fas fa-sign-in-alt me-2"></i>Login
                                    </button>
                                </form>
                                
                                <div class="text-center mt-4">
                                    <small class="text-white-50">Default: admin / admin123</small>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </body>
        </html>
        <?php
        exit;
    }
}

// Handle logout
if (isset($_GET['logout'])) {
    session_destroy();
    header('Location: admin.php');
    exit;
}

// Initialize database and logger
try {
    $database = new Database();
    $logger = new DatabaseLogger($database->getConnection(), 'admin');
} catch (Exception $e) {
    die("Database connection failed: " . $e->getMessage());
}

// Get statistics
function getStats($database) {
    try {
        $pdo = $database->getConnection();
        
        // Get user count
        $stmt = $pdo->query("SELECT COUNT(*) as total_users FROM users");
        $totalUsers = $stmt->fetchColumn();
        
        // Get search count
        $stmt = $pdo->query("SELECT COUNT(*) as total_searches FROM user_searches");
        $totalSearches = $stmt->fetchColumn();
        
        // Get ad clicks
        $stmt = $pdo->query("SELECT COUNT(*) as total_ad_clicks FROM bot_logs WHERE log_type = 'ad_click'");
        $totalAdClicks = $stmt->fetchColumn();
        
        // Get recent activity
        $stmt = $pdo->query("SELECT COUNT(*) as recent_activity FROM user_searches WHERE created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)");
        $recentActivity = $stmt->fetchColumn();
        
        return [
            'total_users' => $totalUsers,
            'total_searches' => $totalSearches,
            'total_ad_clicks' => $totalAdClicks,
            'recent_activity' => $recentActivity
        ];
    } catch (Exception $e) {
        return [
            'total_users' => 0,
            'total_searches' => 0,
            'total_ad_clicks' => 0,
            'recent_activity' => 0
        ];
    }
}

// Get recent logs
function getRecentLogs($database, $limit = 50) {
    try {
        $pdo = $database->getConnection();
        $stmt = $pdo->prepare("
            SELECT * FROM bot_logs 
            ORDER BY created_at DESC 
            LIMIT ?
        ");
        $stmt->execute([$limit]);
        return $stmt->fetchAll();
    } catch (Exception $e) {
        return [];
    }
}

// Get system logs
function getSystemLogs($database, $limit = 50) {
    try {
        $pdo = $database->getConnection();
        $stmt = $pdo->prepare("
            SELECT * FROM system_logs 
            ORDER BY created_at DESC 
            LIMIT ?
        ");
        $stmt->execute([$limit]);
        return $stmt->fetchAll();
    } catch (Exception $e) {
        return [];
    }
}

// Get log settings
function getLogSettings($database) {
    try {
        $pdo = $database->getConnection();
        $stmt = $pdo->query("SELECT * FROM log_settings");
        $settings = [];
        while ($row = $stmt->fetch()) {
            $settings[$row['setting_name']] = $row['setting_value'];
        }
        return $settings;
    } catch (Exception $e) {
        return [];
    }
}

// Handle AJAX requests
if (isset($_GET['action'])) {
    header('Content-Type: application/json');
    
    switch ($_GET['action']) {
        case 'get_stats':
            echo json_encode(getStats($database));
            exit;
            
        case 'get_logs':
            $type = $_GET['type'] ?? 'bot';
            $limit = (int)($_GET['limit'] ?? 50);
            
            if ($type === 'bot') {
                echo json_encode(getRecentLogs($database, $limit));
        } else {
                echo json_encode(getSystemLogs($database, $limit));
            }
            exit;
            
        case 'clear_logs':
            try {
                $pdo = $database->getConnection();
                $stmt = $pdo->prepare("DELETE FROM bot_logs");
                $stmt->execute();
                echo json_encode(['success' => true, 'message' => 'Logs cleared successfully']);
            } catch (Exception $e) {
                echo json_encode(['success' => false, 'message' => $e->getMessage()]);
            }
            exit;
            
        case 'update_settings':
            try {
                $input = json_decode(file_get_contents('php://input'), true);
                $pdo = $database->getConnection();
                
                foreach ($input as $key => $value) {
                    $stmt = $pdo->prepare("UPDATE log_settings SET setting_value = ? WHERE setting_name = ?");
                    $stmt->execute([$value, $key]);
                }
                
                echo json_encode(['success' => true, 'message' => 'Settings updated successfully']);
            } catch (Exception $e) {
                echo json_encode(['success' => false, 'message' => $e->getMessage()]);
            }
            exit;
            
        case 'get_users':
            try {
                $pdo = $database->getConnection();
                $limit = (int)($_GET['limit'] ?? 50);
                $offset = (int)($_GET['offset'] ?? 0);
                
                $stmt = $pdo->prepare("
                    SELECT id, username, first_name, last_name, last_seen, created_at 
                    FROM users 
                    ORDER BY last_seen DESC 
                    LIMIT ? OFFSET ?
                ");
                $stmt->execute([$limit, $offset]);
                $users = $stmt->fetchAll();
                
                // Get total count
                $stmt = $pdo->query("SELECT COUNT(*) as total FROM users");
                $total = $stmt->fetchColumn();
                
                echo json_encode([
                    'success' => true,
                    'users' => $users,
                    'total' => $total,
                    'limit' => $limit,
                    'offset' => $offset
                ]);
            } catch (Exception $e) {
                echo json_encode(['success' => false, 'message' => $e->getMessage()]);
            }
            exit;
            
        case 'get_broadcasts':
            try {
                $pdo = $database->getConnection();
                $limit = (int)($_GET['limit'] ?? 50);
                $offset = (int)($_GET['offset'] ?? 0);
                
                $stmt = $pdo->prepare("
                    SELECT id, message, status, success_count, failed_count, total_users, 
                           created_at, completed_at 
                    FROM broadcast_logs 
                    ORDER BY created_at DESC 
                    LIMIT ? OFFSET ?
                ");
                $stmt->execute([$limit, $offset]);
                $broadcasts = $stmt->fetchAll();
                
                echo json_encode([
                    'success' => true,
                    'broadcasts' => $broadcasts
                ]);
            } catch (Exception $e) {
                echo json_encode(['success' => false, 'message' => $e->getMessage()]);
            }
            exit;
    }
}

$stats = getStats($database);
$recentLogs = getRecentLogs($database, 20);
$systemLogs = getSystemLogs($database, 20);
$logSettings = getLogSettings($database);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Panel - Telegram Bot</title>
    
    <!-- Bootstrap 5 CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    
    <!-- Font Awesome -->
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    
    <!-- Chart.js -->
    <script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>
    
    <!-- Custom CSS -->
    <style>
        :root {
            --primary-color: #667eea;
            --secondary-color: #764ba2;
            --success-color: #10b981;
            --warning-color: #f59e0b;
            --danger-color: #ef4444;
            --info-color: #3b82f6;
            --dark-color: #1f2937;
            --light-color: #f8fafc;
        }
        
        body {
            background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%);
            min-height: 100vh;
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, sans-serif;
        }
        
        .sidebar {
            background: rgba(255, 255, 255, 0.1);
            backdrop-filter: blur(10px);
            border-right: 1px solid rgba(255, 255, 255, 0.2);
            min-height: 100vh;
            position: fixed;
            top: 0;
            left: 0;
            width: 280px;
            z-index: 1000;
            transition: all 0.3s ease;
        }
        
        .sidebar.collapsed {
            width: 80px;
        }
        
        .sidebar-header {
            padding: 1.5rem;
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
        }
        
        .sidebar-nav {
            padding: 1rem 0;
        }
        
        .nav-item {
            margin: 0.25rem 1rem;
        }
        
        .nav-link {
            color: rgba(255, 255, 255, 0.8);
            padding: 0.75rem 1rem;
            border-radius: 0.5rem;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            text-decoration: none;
        }
        
        .nav-link:hover {
            background: rgba(255, 255, 255, 0.1);
            color: white;
            transform: translateX(5px);
        }
        
        .nav-link.active {
            background: rgba(255, 255, 255, 0.2);
            color: white;
        }
        
        .nav-link i {
            width: 20px;
            margin-right: 0.75rem;
        }
        
        .main-content {
            margin-left: 280px;
            padding: 2rem;
            transition: all 0.3s ease;
        }
        
        .main-content.expanded {
            margin-left: 80px;
        }
        
        .card {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            border: 1px solid rgba(255, 255, 255, 0.2);
            border-radius: 1rem;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
            transition: all 0.3s ease;
        }
        
        .card:hover {
            transform: translateY(-2px);
            box-shadow: 0 12px 40px rgba(0, 0, 0, 0.15);
        }
        
        .stat-card {
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            color: white;
            border: none;
        }
        
        .stat-card .card-body {
            padding: 1.5rem;
        }
        
        .stat-number {
            font-size: 2.5rem;
            font-weight: 700;
            margin-bottom: 0.5rem;
        }
        
        .stat-label {
            font-size: 0.875rem;
            opacity: 0.9;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .stat-icon {
            font-size: 2rem;
            opacity: 0.8;
        }
        
        .btn-primary {
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            border: none;
            border-radius: 0.5rem;
            padding: 0.75rem 1.5rem;
            font-weight: 500;
            transition: all 0.3s ease;
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(102, 126, 234, 0.4);
        }
        
        .btn-success {
            background: linear-gradient(135deg, var(--success-color), #059669);
            border: none;
        }
        
        .btn-warning {
            background: linear-gradient(135deg, var(--warning-color), #d97706);
            border: none;
        }
        
        .btn-danger {
            background: linear-gradient(135deg, var(--danger-color), #dc2626);
            border: none;
        }
        
        .btn-info {
            background: linear-gradient(135deg, var(--info-color), #2563eb);
            border: none;
        }
        
        .log-container {
            background: #1a1a1a;
            color: #00ff00;
            font-family: 'Courier New', monospace;
            font-size: 0.875rem;
            border-radius: 0.5rem;
            padding: 1rem;
            max-height: 400px;
            overflow-y: auto;
        }
        
        .log-entry {
            margin-bottom: 0.5rem;
            padding: 0.25rem 0;
        }
        
        .log-timestamp {
            color: #ffd700;
        }
        
        .log-level {
            font-weight: bold;
        }
        
        .log-level.info { color: #00ff00; }
        .log-level.warning { color: #ffaa00; }
        .log-level.error { color: #ff4444; }
        .log-level.debug { color: #8888ff; }
        
        .chart-container {
            position: relative;
            height: 300px;
        }
        
        .table-responsive {
            border-radius: 0.5rem;
            overflow: hidden;
        }
        
        .table {
            margin-bottom: 0;
        }
        
        .table thead th {
            background: rgba(102, 126, 234, 0.1);
            border: none;
            font-weight: 600;
            color: var(--dark-color);
        }
        
        .table tbody tr {
            transition: all 0.3s ease;
        }
        
        .table tbody tr:hover {
            background: rgba(102, 126, 234, 0.05);
        }
        
        .badge {
            font-size: 0.75rem;
            padding: 0.5rem 0.75rem;
            border-radius: 0.375rem;
        }
        
        .sidebar-toggle {
            position: fixed;
            top: 1rem;
            left: 1rem;
            z-index: 1001;
            background: rgba(255, 255, 255, 0.2);
            border: none;
            color: white;
            padding: 0.5rem;
            border-radius: 0.5rem;
            backdrop-filter: blur(10px);
        }
        
        .status-indicator {
            width: 8px;
            height: 8px;
            border-radius: 50%;
            display: inline-block;
            margin-right: 0.5rem;
        }
        
        .status-online { background: var(--success-color); }
        .status-offline { background: var(--danger-color); }
        
        @media (max-width: 768px) {
            .sidebar {
                transform: translateX(-100%);
            }
            
            .sidebar.show {
                transform: translateX(0);
            }
            
            .main-content {
                margin-left: 0;
            }
        }
        
        .fade-in {
            animation: fadeIn 0.5s ease-in;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }
        
        .pulse {
            animation: pulse 2s infinite;
        }
        
        @keyframes pulse {
            0%, 100% { opacity: 1; }
            50% { opacity: 0.5; }
        }
    </style>
</head>
<body>
    <!-- Sidebar Toggle -->
    <button class="sidebar-toggle d-lg-none" onclick="toggleSidebar()">
        <i class="fas fa-bars"></i>
    </button>
    
    <!-- Sidebar -->
    <nav class="sidebar" id="sidebar">
        <div class="sidebar-header">
            <div class="d-flex align-items-center">
                <i class="fas fa-robot fa-2x text-white me-3"></i>
                <div class="d-none d-lg-block">
                    <h5 class="text-white mb-0">Bot Admin</h5>
                    <small class="text-white-50">Management Panel</small>
                </div>
        </div>
    </div>
        
        <div class="sidebar-nav">
            <div class="nav-item">
                <a href="#dashboard" class="nav-link active" onclick="showSection('dashboard')">
                    <i class="fas fa-tachometer-alt"></i>
                    <span class="d-none d-lg-inline">Dashboard</span>
                </a>
                </div>
            <div class="nav-item">
                <a href="#users" class="nav-link" onclick="showSection('users')">
                    <i class="fas fa-users"></i>
                    <span class="d-none d-lg-inline">Users</span>
                </a>
                    </div>
            <div class="nav-item">
                <a href="#logs" class="nav-link" onclick="showSection('logs')">
                    <i class="fas fa-terminal"></i>
                    <span class="d-none d-lg-inline">Logs</span>
                </a>
                    </div>
            <div class="nav-item">
                <a href="#settings" class="nav-link" onclick="showSection('settings')">
                    <i class="fas fa-cog"></i>
                    <span class="d-none d-lg-inline">Settings</span>
                    </a>
                </div>
            <div class="nav-item">
                <a href="#analytics" class="nav-link" onclick="showSection('analytics')">
                    <i class="fas fa-chart-line"></i>
                    <span class="d-none d-lg-inline">Analytics</span>
                </a>
            </div>
            <div class="nav-item">
                <a href="#broadcast" class="nav-link" onclick="showSection('broadcast')">
                    <i class="fas fa-bullhorn"></i>
                    <span class="d-none d-lg-inline">Broadcast</span>
                </a>
        </div>
    </div>

        <div class="mt-auto p-3">
            <div class="d-flex align-items-center text-white-50">
                <span class="status-indicator status-online"></span>
                <small>System Online</small>
                        </div>
            <div class="mt-2">
                <a href="?logout=1" class="btn btn-outline-light btn-sm w-100">
                    <i class="fas fa-sign-out-alt me-2"></i>Logout
                </a>
                    </div>
        </div>
    </nav>
    
    <!-- Main Content -->
    <div class="main-content" id="mainContent">
        <!-- Dashboard Section -->
        <div id="dashboard" class="content-section">
            <div class="d-flex justify-content-between align-items-center mb-4">
                        <div>
                    <h2 class="text-white mb-1">Dashboard</h2>
                    <p class="text-white-50 mb-0">Welcome back, <?= $_SESSION['admin_username'] ?? 'Admin' ?>!</p>
                        </div>
                <div class="d-flex align-items-center">
                    <span class="text-white-50 me-3">
                        <i class="fas fa-clock me-2"></i>
                        <span id="currentTime"></span>
                    </span>
                    <button class="btn btn-outline-light" onclick="refreshData()">
                        <i class="fas fa-sync-alt me-2"></i>Refresh
                        </button>
                    </div>
                </div>
                
            <!-- Stats Cards -->
            <div class="row mb-4">
                <div class="col-xl-3 col-md-6 mb-4">
                    <div class="card stat-card fade-in">
                        <div class="card-body">
                            <div class="d-flex justify-content-between">
                        <div>
                                    <div class="stat-number" id="totalUsers"><?= number_format($stats['total_users']) ?></div>
                                    <div class="stat-label">Total Users</div>
                        </div>
                                <div class="stat-icon">
                                    <i class="fas fa-users"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="col-xl-3 col-md-6 mb-4">
                    <div class="card stat-card fade-in">
                        <div class="card-body">
                            <div class="d-flex justify-content-between">
                        <div>
                                    <div class="stat-number" id="totalSearches"><?= number_format($stats['total_searches']) ?></div>
                                    <div class="stat-label">Total Searches</div>
                        </div>
                                <div class="stat-icon">
                                    <i class="fas fa-search"></i>
                                </div>
                            </div>
                        </div>
                    </div>
        </div>
        
                <div class="col-xl-3 col-md-6 mb-4">
                    <div class="card stat-card fade-in">
                        <div class="card-body">
                            <div class="d-flex justify-content-between">
                        <div>
                                    <div class="stat-number" id="totalAdClicks"><?= number_format($stats['total_ad_clicks']) ?></div>
                                    <div class="stat-label">Ad Clicks</div>
            </div>
                                <div class="stat-icon">
                                    <i class="fas fa-mouse-pointer"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="col-xl-3 col-md-6 mb-4">
                    <div class="card stat-card fade-in">
                        <div class="card-body">
                            <div class="d-flex justify-content-between">
                        <div>
                                    <div class="stat-number" id="recentActivity"><?= number_format($stats['recent_activity']) ?></div>
                                    <div class="stat-label">Recent Activity</div>
                    </div>
                                <div class="stat-icon">
                                    <i class="fas fa-chart-line"></i>
                    </div>
                    </div>
                    </div>
        </div>
            </div>
                </div>
                
            <!-- Charts Row -->
            <div class="row mb-4">
                <div class="col-lg-8 mb-4">
                    <div class="card">
                        <div class="card-header">
                            <h5 class="card-title mb-0">
                                <i class="fas fa-chart-bar me-2"></i>Activity Overview
                            </h5>
                    </div>
                        <div class="card-body">
                            <div class="chart-container">
                                <canvas id="activityChart"></canvas>
                    </div>
                    </div>
                    </div>
            </div>
                
                <div class="col-lg-4 mb-4">
                    <div class="card">
                        <div class="card-header">
                            <h5 class="card-title mb-0">
                                <i class="fas fa-pie-chart me-2"></i>Distribution
                            </h5>
                        </div>
                        <div class="card-body">
                            <div class="chart-container">
                                <canvas id="distributionChart"></canvas>
                    </div>
                </div>
            </div>
                        </div>
                </div>
                
            <!-- Recent Activity -->
            <div class="row">
                <div class="col-12">
                    <div class="card">
                        <div class="card-header">
                            <h5 class="card-title mb-0">
                                <i class="fas fa-history me-2"></i>Recent Activity
                            </h5>
                    </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th>Time</th>
                                            <th>Type</th>
                                            <th>User</th>
                                            <th>Action</th>
                                            <th>Details</th>
                                        </tr>
                                    </thead>
                                    <tbody id="recentActivityTable">
                                        <?php foreach (array_slice($recentLogs, 0, 10) as $log): ?>
                                        <tr>
                                            <td><?= date('H:i:s', strtotime($log['created_at'])) ?></td>
                                            <td>
                                                <span class="badge bg-<?= $log['log_type'] === 'ad_click' ? 'success' : 'info' ?>">
                                                    <?= ucfirst($log['log_type']) ?>
                                                </span>
                                            </td>
                                            <td><?= $log['username'] ?? 'N/A' ?></td>
                                            <td><?= $log['action'] ?? 'N/A' ?></td>
                                            <td><?= substr($log['details'] ?? '', 0, 50) ?>...</td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                    </div>
                    </div>
                        </div>
                    </div>
                                </div>
                                    </div>
        
        <!-- Users Section -->
        <div id="users" class="content-section d-none">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h2 class="text-white">Users Management</h2>
                <button class="btn btn-primary" onclick="refreshUsers()">
                    <i class="fas fa-sync-alt me-2"></i>Refresh
                </button>
                                </div>
            
            <div class="card">
                <div class="card-header">
                    <h5 class="card-title mb-0">
                        <i class="fas fa-users me-2"></i>Registered Users
                    </h5>
                                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Username</th>
                                    <th>Name</th>
                                    <th>Last Seen</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody id="usersTable">
                                <!-- Users will be loaded here -->
                            </tbody>
                        </table>
                            </div>
                        </div>
                        </div>
                        </div>
        
        <!-- Logs Section -->
        <div id="logs" class="content-section d-none">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h2 class="text-white">System Logs</h2>
                <div class="btn-group">
                    <button class="btn btn-outline-light" onclick="refreshLogs()">
                        <i class="fas fa-sync-alt me-2"></i>Refresh
                    </button>
                    <button class="btn btn-danger" onclick="clearLogs()">
                        <i class="fas fa-trash me-2"></i>Clear
                    </button>
                </div>
                </div>
                
            <div class="row">
                <div class="col-lg-6 mb-4">
                    <div class="card">
                        <div class="card-header">
                            <h5 class="card-title mb-0">
                                <i class="fas fa-robot me-2"></i>Bot Logs
                            </h5>
                    </div>
                        <div class="card-body">
                            <div class="log-container" id="botLogs">
                                <?php foreach (array_slice($recentLogs, 0, 20) as $log): ?>
                                <div class="log-entry">
                                    <span class="log-timestamp">[<?= date('H:i:s', strtotime($log['created_at'])) ?>]</span>
                                    <span class="log-level <?= $log['log_type'] ?>">[<?= strtoupper($log['log_type']) ?>]</span>
                                    <span><?= $log['message'] ?></span>
                </div>
                                <?php endforeach; ?>
                    </div>
                </div>
                    </div>
                </div>
                
                <div class="col-lg-6 mb-4">
                    <div class="card">
                        <div class="card-header">
                            <h5 class="card-title mb-0">
                                <i class="fas fa-cog me-2"></i>System Logs
                            </h5>
                    </div>
                        <div class="card-body">
                            <div class="log-container" id="systemLogs">
                                <?php foreach (array_slice($systemLogs, 0, 20) as $log): ?>
                                <div class="log-entry">
                                    <span class="log-timestamp">[<?= date('H:i:s', strtotime($log['created_at'])) ?>]</span>
                                    <span class="log-level <?= strtolower($log['level']) ?>">[<?= $log['level'] ?>]</span>
                                    <span><?= $log['message'] ?></span>
                    </div>
                                <?php endforeach; ?>
                </div>
                </div>
                    </div>
                </div>
                        </div>
                        </div>
        
        <!-- Settings Section -->
        <div id="settings" class="content-section d-none">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h2 class="text-white">System Settings</h2>
                <button class="btn btn-primary" onclick="saveSettings()">
                    <i class="fas fa-save me-2"></i>Save Settings
                            </button>
                    </div>
                    
            <div class="row">
                <div class="col-lg-6 mb-4">
                    <div class="card">
                        <div class="card-header">
                            <h5 class="card-title mb-0">
                                <i class="fas fa-cog me-2"></i>Log Settings
                            </h5>
                                    </div>
                        <div class="card-body">
                            <form id="settingsForm">
                                <div class="mb-3">
                                    <label class="form-label">Auto Reset (hours)</label>
                                    <input type="number" class="form-control" name="auto_reset_hours" 
                                           value="<?= $logSettings['auto_reset_hours'] ?? '24' ?>" min="0" max="168">
                                </div>
                                <div class="mb-3">
                                    <label class="form-label">Max Log Entries</label>
                                    <input type="number" class="form-control" name="max_log_entries" 
                                           value="<?= $logSettings['max_log_entries'] ?? '10000' ?>" min="100" max="100000">
                            </div>
                                <div class="mb-3">
                                    <label class="form-label">Retention Days</label>
                                    <input type="number" class="form-control" name="log_retention_days" 
                                           value="<?= $logSettings['log_retention_days'] ?? '30' ?>" min="1" max="365">
                            </div>
                                <div class="mb-3">
                                    <label class="form-label">Log Level</label>
                                    <select class="form-select" name="log_level">
                                        <option value="DEBUG" <?= ($logSettings['log_level'] ?? 'INFO') === 'DEBUG' ? 'selected' : '' ?>>DEBUG</option>
                                        <option value="INFO" <?= ($logSettings['log_level'] ?? 'INFO') === 'INFO' ? 'selected' : '' ?>>INFO</option>
                                        <option value="WARNING" <?= ($logSettings['log_level'] ?? 'INFO') === 'WARNING' ? 'selected' : '' ?>>WARNING</option>
                                        <option value="ERROR" <?= ($logSettings['log_level'] ?? 'INFO') === 'ERROR' ? 'selected' : '' ?>>ERROR</option>
                                        <option value="CRITICAL" <?= ($logSettings['log_level'] ?? 'INFO') === 'CRITICAL' ? 'selected' : '' ?>>CRITICAL</option>
                                    </select>
                    </div>
                            </form>
                        </div>
                    </div>
                </div>
                
                <div class="col-lg-6 mb-4">
                    <div class="card">
                        <div class="card-header">
                            <h5 class="card-title mb-0">
                                <i class="fas fa-info-circle me-2"></i>System Info
                            </h5>
                        </div>
                        <div class="card-body">
                            <div class="row">
                                <div class="col-6">
                                    <div class="text-center">
                                        <div class="h4 text-primary"><?= PHP_VERSION ?></div>
                                        <small class="text-muted">PHP Version</small>
                        </div>
                            </div>
                                <div class="col-6">
                                    <div class="text-center">
                                        <div class="h4 text-success"><?= date('Y-m-d') ?></div>
                                        <small class="text-muted">Last Update</small>
                            </div>
                        </div>
                            </div>
                        </div>
                    </div>
                            </div>
                </div>
                </div>
                
        <!-- Analytics Section -->
        <div id="analytics" class="content-section d-none">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h2 class="text-white">Analytics</h2>
                <button class="btn btn-primary" onclick="refreshAnalytics()">
                    <i class="fas fa-sync-alt me-2"></i>Refresh
                </button>
                </div>
                
            <div class="row">
                <div class="col-12">
                    <div class="card">
                        <div class="card-header">
                            <h5 class="card-title mb-0">
                                <i class="fas fa-chart-line me-2"></i>Performance Metrics
                            </h5>
                        </div>
                        <div class="card-body">
                            <div class="chart-container">
                                <canvas id="performanceChart"></canvas>
                        </div>
                        </div>
                    </div>
                        </div>
                </div>
                </div>
                
        <!-- Broadcast Section -->
        <div id="broadcast" class="content-section d-none">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h2 class="text-white">Broadcast Messages</h2>
                <button class="btn btn-primary" onclick="showBroadcastModal()">
                    <i class="fas fa-plus me-2"></i>New Broadcast
                </button>
                </div>
                
            <div class="card">
                <div class="card-header">
                    <h5 class="card-title mb-0">
                        <i class="fas fa-bullhorn me-2"></i>Broadcast History
                    </h5>
                            </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Message</th>
                                    <th>Status</th>
                                    <th>Sent</th>
                                    <th>Failed</th>
                                    <th>Created</th>
                                </tr>
                            </thead>
                            <tbody id="broadcastTable">
                                <!-- Broadcast history will be loaded here -->
                            </tbody>
                        </table>
                                    </div>
                                </div>
                                </div>
                            </div>
                    </div>
                        
    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    
    <!-- Custom JavaScript -->
    <script>
        // Global variables
        let charts = {};
        let refreshInterval;
        
        // Initialize on page load
        document.addEventListener('DOMContentLoaded', function() {
            initializeCharts();
            updateTime();
            setInterval(updateTime, 1000);
            startAutoRefresh();
        });
        
        // Sidebar toggle
        function toggleSidebar() {
            const sidebar = document.getElementById('sidebar');
            const mainContent = document.getElementById('mainContent');
            
            if (window.innerWidth <= 768) {
                sidebar.classList.toggle('show');
            } else {
                sidebar.classList.toggle('collapsed');
                mainContent.classList.toggle('expanded');
            }
        }
        
        // Navigation
        function showSection(sectionName) {
            // Hide all sections
            document.querySelectorAll('.content-section').forEach(section => {
                section.classList.add('d-none');
            });
            
            // Show selected section
            document.getElementById(sectionName).classList.remove('d-none');
            
            // Update active nav
            document.querySelectorAll('.nav-link').forEach(link => {
                link.classList.remove('active');
            });
            event.target.classList.add('active');
            
            // Load section-specific data
            switch(sectionName) {
                case 'dashboard':
                    refreshData();
                    break;
                case 'users':
                    loadUsers();
                    break;
                case 'logs':
                    refreshLogs();
                    break;
                case 'analytics':
                    refreshAnalytics();
                    break;
                case 'broadcast':
                    loadBroadcasts();
                    break;
            }
        }
        
        // Initialize charts
        function initializeCharts() {
            // Activity Chart
            const activityCtx = document.getElementById('activityChart');
            if (activityCtx) {
                charts.activity = new Chart(activityCtx, {
                    type: 'line',
                    data: {
                        labels: ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'],
                        datasets: [{
                            label: 'Searches',
                            data: [12, 19, 3, 5, 2, 3, 8],
                            borderColor: 'rgb(102, 126, 234)',
                            backgroundColor: 'rgba(102, 126, 234, 0.1)',
                            tension: 0.4
                        }, {
                            label: 'Ad Clicks',
                            data: [2, 3, 20, 5, 1, 4, 6],
                            borderColor: 'rgb(16, 185, 129)',
                            backgroundColor: 'rgba(16, 185, 129, 0.1)',
                            tension: 0.4
                        }]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: {
                            legend: {
                                position: 'top',
                            }
                        },
                        scales: {
                            y: {
                                beginAtZero: true
                            }
                        }
                    }
                });
            }
            
            // Distribution Chart
            const distributionCtx = document.getElementById('distributionChart');
            if (distributionCtx) {
                charts.distribution = new Chart(distributionCtx, {
                    type: 'doughnut',
                    data: {
                        labels: ['Searches', 'Ad Clicks'],
                        datasets: [{
                            data: [<?= $stats['total_searches'] ?>, <?= $stats['total_ad_clicks'] ?>],
                            backgroundColor: [
                                'rgba(102, 126, 234, 0.8)',
                                'rgba(16, 185, 129, 0.8)'
                            ],
                            borderColor: [
                                'rgba(102, 126, 234, 1)',
                                'rgba(16, 185, 129, 1)'
                            ],
                            borderWidth: 2
                        }]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: {
                            legend: {
                                position: 'bottom'
                            }
                        }
                    }
                });
            }
        }
        
        // Update time
        function updateTime() {
            const now = new Date();
            document.getElementById('currentTime').textContent = now.toLocaleTimeString();
        }
        
        // Refresh data
        function refreshData() {
            fetch('?action=get_stats')
                .then(response => response.json())
                .then(data => {
                    document.getElementById('totalUsers').textContent = data.total_users.toLocaleString();
                    document.getElementById('totalSearches').textContent = data.total_searches.toLocaleString();
                    document.getElementById('totalAdClicks').textContent = data.total_ad_clicks.toLocaleString();
                    document.getElementById('recentActivity').textContent = data.recent_activity.toLocaleString();
                })
                .catch(error => console.error('Error:', error));
        }
        
        // Refresh logs
        function refreshLogs() {
            // Bot logs
            fetch('?action=get_logs&type=bot&limit=20')
                .then(response => response.json())
                .then(data => {
                    const container = document.getElementById('botLogs');
                    container.innerHTML = data.map(log => `
                        <div class="log-entry">
                            <span class="log-timestamp">[${new Date(log.created_at).toLocaleTimeString()}]</span>
                            <span class="log-level ${log.log_type}">[${log.log_type.toUpperCase()}]</span>
                            <span>${log.message}</span>
                        </div>
                    `).join('');
                })
                .catch(error => console.error('Error:', error));
            
            // System logs
            fetch('?action=get_logs&type=system&limit=20')
                .then(response => response.json())
                .then(data => {
                    const container = document.getElementById('systemLogs');
                    container.innerHTML = data.map(log => `
                        <div class="log-entry">
                            <span class="log-timestamp">[${new Date(log.created_at).toLocaleTimeString()}]</span>
                            <span class="log-level ${log.level.toLowerCase()}">[${log.level}]</span>
                            <span>${log.message}</span>
                        </div>
                    `).join('');
                })
                .catch(error => console.error('Error:', error));
        }
        
        // Clear logs
        function clearLogs() {
            if (confirm('Are you sure you want to clear all logs? This action cannot be undone.')) {
                fetch('?action=clear_logs', { method: 'POST' })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            alert('Logs cleared successfully');
                            refreshLogs();
                        } else {
                            alert('Error: ' + data.message);
                        }
                    })
                    .catch(error => console.error('Error:', error));
            }
        }
        
        // Save settings
        function saveSettings() {
            const form = document.getElementById('settingsForm');
            const formData = new FormData(form);
            const settings = {};
            
            for (let [key, value] of formData.entries()) {
                settings[key] = value;
            }
            
            fetch('?action=update_settings', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify(settings)
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert('Settings saved successfully');
            } else {
                    alert('Error: ' + data.message);
                }
            })
            .catch(error => console.error('Error:', error));
        }
        
        // Load users
        function loadUsers() {
            const tableBody = document.getElementById('usersTable');
            tableBody.innerHTML = '<tr><td colspan="5" class="text-center">Loading users...</td></tr>';
            
            fetch('?action=get_users&limit=50')
                .then(response => response.json())
                .then(data => {
                    if (data.success && data.users) {
                        if (data.users.length === 0) {
                            tableBody.innerHTML = '<tr><td colspan="5" class="text-center text-muted">No users found</td></tr>';
                        } else {
                            tableBody.innerHTML = data.users.map(user => `
                                <tr>
                                    <td>${user.id}</td>
                                    <td>${user.username || 'N/A'}</td>
                                    <td>${user.first_name || ''} ${user.last_name || ''}</td>
                                    <td>${user.last_seen ? new Date(user.last_seen).toLocaleString() : 'Never'}</td>
                                    <td>
                                        <button class="btn btn-sm btn-outline-primary" onclick="viewUser(${user.id})">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                        <button class="btn btn-sm btn-outline-danger" onclick="deleteUser(${user.id})">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </td>
                                </tr>
                            `).join('');
                        }
                    } else {
                        tableBody.innerHTML = '<tr><td colspan="5" class="text-center text-danger">Error loading users: ' + (data.message || 'Unknown error') + '</td></tr>';
                    }
                })
                .catch(error => {
                    console.error('Error loading users:', error);
                    tableBody.innerHTML = '<tr><td colspan="5" class="text-center text-danger">Error loading users</td></tr>';
                });
        }
        
        // Refresh users
        function refreshUsers() {
            loadUsers();
        }
        
        // View user details
        function viewUser(userId) {
            alert('View user details for ID: ' + userId);
            // You can implement a modal or redirect to user details page
        }
        
        // Delete user
        function deleteUser(userId) {
            if (confirm('Are you sure you want to delete this user?')) {
                // Implement delete functionality
                alert('Delete user ID: ' + userId);
            }
        }
        
        // Refresh analytics
        function refreshAnalytics() {
            // This would typically fetch analytics data
            console.log('Refreshing analytics...');
        }
        
        // Load broadcasts
        function loadBroadcasts() {
            const tableBody = document.getElementById('broadcastTable');
            tableBody.innerHTML = '<tr><td colspan="6" class="text-center">Loading broadcasts...</td></tr>';
            
            fetch('?action=get_broadcasts&limit=50')
                .then(response => response.json())
                .then(data => {
                    if (data.success && data.broadcasts) {
                        if (data.broadcasts.length === 0) {
                            tableBody.innerHTML = '<tr><td colspan="6" class="text-center text-muted">No broadcasts found</td></tr>';
                        } else {
                            tableBody.innerHTML = data.broadcasts.map(broadcast => `
                                <tr>
                                    <td>${broadcast.id}</td>
                                    <td>${broadcast.message.length > 50 ? broadcast.message.substring(0, 50) + '...' : broadcast.message}</td>
                                    <td>
                                        <span class="badge bg-${broadcast.status === 'completed' ? 'success' : broadcast.status === 'running' ? 'warning' : 'secondary'}">
                                            ${broadcast.status}
                                        </span>
                                    </td>
                                    <td>${broadcast.success_count || 0}</td>
                                    <td>${broadcast.failed_count || 0}</td>
                                    <td>${broadcast.created_at ? new Date(broadcast.created_at).toLocaleString() : 'N/A'}</td>
                                </tr>
                            `).join('');
                        }
                    } else {
                        tableBody.innerHTML = '<tr><td colspan="6" class="text-center text-danger">Error loading broadcasts: ' + (data.message || 'Unknown error') + '</td></tr>';
                    }
                })
                .catch(error => {
                    console.error('Error loading broadcasts:', error);
                    tableBody.innerHTML = '<tr><td colspan="6" class="text-center text-danger">Error loading broadcasts</td></tr>';
                });
        }
        
        // Show broadcast modal
        function showBroadcastModal() {
            alert('Broadcast modal would open here');
        }
        
        // Auto refresh
        function startAutoRefresh() {
            refreshInterval = setInterval(() => {
                if (!document.querySelector('.content-section.d-none')) {
                    refreshData();
                }
            }, 30000); // Refresh every 30 seconds
        }
        
        // Stop auto refresh
        function stopAutoRefresh() {
            if (refreshInterval) {
                clearInterval(refreshInterval);
            }
        }
        
        // Handle window resize
        window.addEventListener('resize', function() {
            if (window.innerWidth > 768) {
                document.getElementById('sidebar').classList.remove('show');
            }
        });
    </script>
</body>
</html>