// Admin Panel JavaScript

// Global variables
let charts = {};
let refreshInterval;

// Initialize on page load
document.addEventListener('DOMContentLoaded', function() {
    initializeCharts();
    updateTime();
    setInterval(updateTime, 1000);
    startAutoRefresh();
    
    // Load data for current section
    const currentSection = getCurrentSection();
    loadSectionData(currentSection);
});

// Get current section from URL
function getCurrentSection() {
    const urlParams = new URLSearchParams(window.location.search);
    return urlParams.get('section') || 'dashboard';
}

// Load data for specific section
function loadSectionData(sectionName) {
    switch(sectionName) {
        case 'dashboard':
            refreshData();
            break;
        case 'users':
            loadUsers();
            break;
        case 'logs':
            refreshLogs();
            break;
        case 'analytics':
            refreshAnalytics();
            break;
        case 'broadcast':
            loadBroadcasts();
            break;
        case 'bot_management':
            if (typeof loadBotStatus === 'function') {
                loadBotStatus();
            }
            break;
    }
}

// Sidebar toggle
function toggleSidebar() {
    const sidebar = document.getElementById('sidebar');
    const mainContent = document.getElementById('mainContent');
    
    if (window.innerWidth <= 768) {
        sidebar.classList.toggle('show');
    } else {
        sidebar.classList.toggle('collapsed');
        mainContent.classList.toggle('expanded');
    }
}

// Update time
function updateTime() {
    const now = new Date();
    const timeElement = document.getElementById('currentTime');
    if (timeElement) {
        timeElement.textContent = now.toLocaleTimeString();
    }
}

// Refresh data
function refreshData() {
    fetch('?action=get_stats')
        .then(response => response.json())
        .then(data => {
            if (data.success === false) {
                console.warn('Database not available:', data.message);
                return;
            }
            
            if (data.total_users !== undefined) {
                const element = document.getElementById('totalUsers');
                if (element) element.textContent = data.total_users.toLocaleString();
            }
            if (data.total_searches !== undefined) {
                const element = document.getElementById('totalSearches');
                if (element) element.textContent = data.total_searches.toLocaleString();
            }
            if (data.total_ad_clicks !== undefined) {
                const element = document.getElementById('totalAdClicks');
                if (element) element.textContent = data.total_ad_clicks.toLocaleString();
            }
            if (data.recent_activity !== undefined) {
                const element = document.getElementById('recentActivity');
                if (element) element.textContent = data.recent_activity.toLocaleString();
            }
        })
        .catch(error => {
            console.error('Error:', error);
            // Show fallback data
            const elements = ['totalUsers', 'totalSearches', 'totalAdClicks', 'recentActivity'];
            elements.forEach(id => {
                const element = document.getElementById(id);
                if (element && element.textContent === '0') {
                    element.textContent = 'N/A';
                }
            });
        });
}

// Refresh logs
function refreshLogs() {
    // Bot logs
    fetch('?action=get_logs&type=bot&limit=20')
        .then(response => response.json())
        .then(data => {
            const container = document.getElementById('botLogs');
            if (container) {
                if (data.success === false) {
                    container.innerHTML = '<div class="log-entry text-warning">Database not available - Logs cannot be loaded</div>';
                    return;
                }
                container.innerHTML = data.map(log => `
                    <div class="log-entry">
                        <span class="log-timestamp">[${new Date(log.created_at).toLocaleTimeString()}]</span>
                        <span class="log-level ${log.log_type}">[${log.log_type.toUpperCase()}]</span>
                        <span>${log.message}</span>
                    </div>
                `).join('');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            const container = document.getElementById('botLogs');
            if (container) {
                container.innerHTML = '<div class="log-entry text-danger">Error loading logs</div>';
            }
        });
    
    // System logs
    fetch('?action=get_logs&type=system&limit=20')
        .then(response => response.json())
        .then(data => {
            const container = document.getElementById('systemLogs');
            if (container) {
                if (data.success === false) {
                    container.innerHTML = '<div class="log-entry text-warning">Database not available - Logs cannot be loaded</div>';
                    return;
                }
                container.innerHTML = data.map(log => `
                    <div class="log-entry">
                        <span class="log-timestamp">[${new Date(log.created_at).toLocaleTimeString()}]</span>
                        <span class="log-level ${log.level.toLowerCase()}">[${log.level}]</span>
                        <span>${log.message}</span>
                    </div>
                `).join('');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            const container = document.getElementById('systemLogs');
            if (container) {
                container.innerHTML = '<div class="log-entry text-danger">Error loading logs</div>';
            }
        });
}

// Clear logs
function clearLogs() {
    if (confirm('Are you sure you want to clear all logs? This action cannot be undone.')) {
        fetch('?action=clear_logs', { method: 'POST' })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert('Logs cleared successfully');
                    refreshLogs();
                } else {
                    alert('Error: ' + data.message);
                }
            })
            .catch(error => console.error('Error:', error));
    }
}

// Save settings
function saveSettings() {
    const form = document.getElementById('settingsForm');
    const formData = new FormData(form);
    const settings = {};
    
    for (let [key, value] of formData.entries()) {
        settings[key] = value;
    }
    
    fetch('?action=update_settings', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify(settings)
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            alert('Settings saved successfully');
        } else {
            alert('Error: ' + data.message);
        }
    })
    .catch(error => console.error('Error:', error));
}

// Load users
function loadUsers() {
    const tableBody = document.getElementById('usersTable');
    if (!tableBody) return;
    
    tableBody.innerHTML = '<tr><td colspan="5" class="text-center">Loading users...</td></tr>';
    
    fetch('?action=get_users&limit=50')
        .then(response => response.json())
        .then(data => {
            if (data.success === false) {
                tableBody.innerHTML = '<tr><td colspan="5" class="text-center text-warning">Database not available - Users cannot be loaded</td></tr>';
                return;
            }
            
            if (data.success && data.users) {
                if (data.users.length === 0) {
                    tableBody.innerHTML = '<tr><td colspan="5" class="text-center text-muted">No users found</td></tr>';
                } else {
                    tableBody.innerHTML = data.users.map(user => `
                        <tr>
                            <td>${user.id}</td>
                            <td>${user.username || 'N/A'}</td>
                            <td>${user.first_name || ''} ${user.last_name || ''}</td>
                            <td>${user.last_seen ? new Date(user.last_seen).toLocaleString() : 'Never'}</td>
                            <td>
                                <button class="btn btn-sm btn-outline-primary" onclick="viewUser(${user.id})">
                                    <i class="fas fa-eye"></i>
                                </button>
                                <button class="btn btn-sm btn-outline-danger" onclick="deleteUser(${user.id})">
                                    <i class="fas fa-trash"></i>
                                </button>
                            </td>
                        </tr>
                    `).join('');
                }
            } else {
                tableBody.innerHTML = '<tr><td colspan="5" class="text-center text-danger">Error loading users: ' + (data.message || 'Unknown error') + '</td></tr>';
            }
        })
        .catch(error => {
            console.error('Error loading users:', error);
            tableBody.innerHTML = '<tr><td colspan="5" class="text-center text-danger">Error loading users</td></tr>';
        });
}

// Refresh users
function refreshUsers() {
    loadUsers();
}

// View user details
function viewUser(userId) {
    alert('View user details for ID: ' + userId);
    // You can implement a modal or redirect to user details page
}

// Delete user
function deleteUser(userId) {
    if (confirm('Are you sure you want to delete this user?')) {
        // Implement delete functionality
        alert('Delete user ID: ' + userId);
    }
}

// Refresh analytics
function refreshAnalytics() {
    console.log('Refreshing analytics...');
    
    // Load stats data
    fetch('?action=get_stats')
        .then(response => response.json())
        .then(data => {
            if (data.success === false) {
                console.warn('Database not available:', data.message);
                return;
            }
            
            // Update analytics stats
            const elements = {
                'totalUsersAnalytics': data.total_users,
                'totalSearchesAnalytics': data.total_searches,
                'totalAdClicksAnalytics': data.total_ad_clicks,
                'recentActivityAnalytics': data.recent_activity
            };
            
            Object.keys(elements).forEach(id => {
                const element = document.getElementById(id);
                if (element) {
                    element.textContent = elements[id].toLocaleString();
                }
            });
            
            // Initialize or update performance chart
            initializePerformanceChart(data);
        })
        .catch(error => {
            console.error('Error loading analytics:', error);
            // Show fallback data
            const elements = ['totalUsersAnalytics', 'totalSearchesAnalytics', 'totalAdClicksAnalytics', 'recentActivityAnalytics'];
            elements.forEach(id => {
                const element = document.getElementById(id);
                if (element) {
                    element.textContent = 'N/A';
                }
            });
        });
}

// Load broadcasts
function loadBroadcasts() {
    const tableBody = document.getElementById('broadcastTable');
    if (!tableBody) return;
    
    tableBody.innerHTML = '<tr><td colspan="7" class="text-center">Loading broadcasts...</td></tr>';
    
    fetch('?action=get_broadcasts&limit=50')
        .then(response => response.json())
        .then(data => {
            if (data.success && data.broadcasts) {
                if (data.broadcasts.length === 0) {
                    tableBody.innerHTML = '<tr><td colspan="7" class="text-center text-muted">No broadcasts found</td></tr>';
                } else {
                    tableBody.innerHTML = data.broadcasts.map(broadcast => `
                        <tr>
                            <td>${broadcast.id}</td>
                            <td>${broadcast.message.length > 50 ? broadcast.message.substring(0, 50) + '...' : broadcast.message}</td>
                            <td>
                                <span class="badge bg-${broadcast.status === 'completed' ? 'success' : broadcast.status === 'running' ? 'warning' : 'secondary'}">
                                    ${broadcast.status}
                                </span>
                            </td>
                            <td>${broadcast.success_count || 0}</td>
                            <td>${broadcast.failed_count || 0}</td>
                            <td>${broadcast.created_at ? new Date(broadcast.created_at).toLocaleString() : 'N/A'}</td>
                            <td>
                                <button class="btn btn-sm btn-outline-info" onclick="viewBroadcast(${broadcast.id})">
                                    <i class="fas fa-eye"></i>
                                </button>
                            </td>
                        </tr>
                    `).join('');
                }
            } else {
                tableBody.innerHTML = '<tr><td colspan="7" class="text-center text-danger">Error loading broadcasts: ' + (data.message || 'Unknown error') + '</td></tr>';
            }
        })
        .catch(error => {
            console.error('Error loading broadcasts:', error);
            tableBody.innerHTML = '<tr><td colspan="7" class="text-center text-danger">Error loading broadcasts</td></tr>';
        });
}

// View broadcast details
function viewBroadcast(broadcastId) {
    alert('View broadcast details for ID: ' + broadcastId);
}

// Show broadcast modal
function showBroadcastModal() {
    const modal = new bootstrap.Modal(document.getElementById('newBroadcastModal'));
    modal.show();
}

// Send broadcast
function sendBroadcast() {
    const form = document.getElementById('broadcastForm');
    const formData = new FormData(form);
    const data = {};
    
    for (let [key, value] of formData.entries()) {
        data[key] = value;
    }
    
    // Validate message
    if (!data.message || data.message.trim() === '') {
        alert('Please enter a message');
        return;
    }
    
    // Show loading
    const sendBtn = document.querySelector('#newBroadcastModal .btn-primary');
    const originalText = sendBtn.innerHTML;
    sendBtn.innerHTML = '<span class="loading"></span> Sending...';
    sendBtn.disabled = true;
    
    fetch('?action=create_broadcast', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify(data)
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            alert('Broadcast created successfully');
            bootstrap.Modal.getInstance(document.getElementById('newBroadcastModal')).hide();
            loadBroadcasts();
        } else {
            alert('Error: ' + data.message);
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Error creating broadcast');
    })
    .finally(() => {
        sendBtn.innerHTML = originalText;
        sendBtn.disabled = false;
    });
}

// Initialize charts
function initializeCharts() {
    // Activity Chart
    const activityCtx = document.getElementById('activityChart');
    if (activityCtx) {
        charts.activity = new Chart(activityCtx, {
            type: 'line',
            data: {
                labels: ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'],
                datasets: [{
                    label: 'Searches',
                    data: [12, 19, 3, 5, 2, 3, 8],
                    borderColor: 'rgb(102, 126, 234)',
                    backgroundColor: 'rgba(102, 126, 234, 0.1)',
                    tension: 0.4
                }, {
                    label: 'Ad Clicks',
                    data: [2, 3, 20, 5, 1, 4, 6],
                    borderColor: 'rgb(16, 185, 129)',
                    backgroundColor: 'rgba(16, 185, 129, 0.1)',
                    tension: 0.4
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'top',
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true
                    }
                }
            }
        });
    }
    
    // Distribution Chart
    const distributionCtx = document.getElementById('distributionChart');
    if (distributionCtx) {
        charts.distribution = new Chart(distributionCtx, {
            type: 'doughnut',
            data: {
                labels: ['Searches', 'Ad Clicks'],
                datasets: [{
                    data: [0, 0],
                    backgroundColor: [
                        'rgba(102, 126, 234, 0.8)',
                        'rgba(16, 185, 129, 0.8)'
                    ],
                    borderColor: [
                        'rgba(102, 126, 234, 1)',
                        'rgba(16, 185, 129, 1)'
                    ],
                    borderWidth: 2
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'bottom'
                    }
                }
            }
        });
    }
}

// Initialize performance chart
function initializePerformanceChart(data) {
    const performanceCtx = document.getElementById('performanceChart');
    if (!performanceCtx) return;
    
    if (charts.performance) {
        // Update existing chart
        charts.performance.data.datasets[0].data = [
            data.total_users || 0,
            data.total_searches || 0,
            data.total_ad_clicks || 0,
            data.recent_activity || 0
        ];
        charts.performance.update();
    } else {
        // Create new chart
        charts.performance = new Chart(performanceCtx, {
            type: 'bar',
            data: {
                labels: ['Total Users', 'Total Searches', 'Ad Clicks', 'Recent Activity'],
                datasets: [{
                    label: 'Count',
                    data: [
                        data.total_users || 0,
                        data.total_searches || 0,
                        data.total_ad_clicks || 0,
                        data.recent_activity || 0
                    ],
                    backgroundColor: [
                        'rgba(102, 126, 234, 0.8)',
                        'rgba(16, 185, 129, 0.8)',
                        'rgba(245, 158, 11, 0.8)',
                        'rgba(59, 130, 246, 0.8)'
                    ],
                    borderColor: [
                        'rgba(102, 126, 234, 1)',
                        'rgba(16, 185, 129, 1)',
                        'rgba(245, 158, 11, 1)',
                        'rgba(59, 130, 246, 1)'
                    ],
                    borderWidth: 2
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true
                    }
                }
            }
        });
    }
}

// Auto refresh
function startAutoRefresh() {
    refreshInterval = setInterval(() => {
        const currentSection = window.location.search.includes('section=') ? 
            window.location.search.split('section=')[1].split('&')[0] : 'dashboard';
        
        if (currentSection === 'dashboard') {
            refreshData();
        }
    }, 30000); // Refresh every 30 seconds
}

// Stop auto refresh
function stopAutoRefresh() {
    if (refreshInterval) {
        clearInterval(refreshInterval);
    }
}

// Handle window resize
window.addEventListener('resize', function() {
    if (window.innerWidth > 768) {
        document.getElementById('sidebar').classList.remove('show');
    }
});

// Handle target type change
document.addEventListener('DOMContentLoaded', function() {
    const targetTypeSelect = document.getElementById('targetType');
    const specificUsersDiv = document.getElementById('specificUsersDiv');
    
    if (targetTypeSelect && specificUsersDiv) {
        targetTypeSelect.addEventListener('change', function() {
            if (this.value === 'specific') {
                specificUsersDiv.style.display = 'block';
            } else {
                specificUsersDiv.style.display = 'none';
            }
        });
    }
});