// Bot Management JavaScript

// Show bot error message
function showBotError(message) {
    const statusIndicator = document.getElementById('botStatusIndicator');
    const statusText = document.getElementById('botStatusText');
    const lastCheck = document.getElementById('botLastCheck');
    const commandsTable = document.getElementById('botCommandsTable');
    const logsContainer = document.getElementById('botLogsContainer');
    
    if (statusIndicator && statusText) {
        statusIndicator.className = 'status-indicator status-offline';
        statusText.textContent = 'Bot manager not available';
    }
    
    if (lastCheck) {
        lastCheck.textContent = 'Error';
    }
    
    if (commandsTable) {
        commandsTable.innerHTML = `
            <tr>
                <td colspan="4" class="text-center text-muted">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    ${message}
                </td>
            </tr>
        `;
    }
    
    if (logsContainer) {
        logsContainer.innerHTML = `
            <div class="text-center text-muted">
                <i class="fas fa-exclamation-triangle me-2"></i>
                ${message}
            </div>
        `;
    }
}

// Load bot status
function loadBotStatus() {
    fetch('?action=get_bot_status')
        .then(response => response.json())
        .then(data => {
            if (data.success === false) {
                console.warn('Bot manager not available:', data.message);
                showBotError(data.message);
                return;
            }
            
            if (data.success) {
                updateBotStatusDisplay(data);
                loadBotCommands(data.commands);
                loadBotLogs(data.recent_logs);
            } else {
                console.error('Error loading bot status:', data.message);
                showBotError(data.message);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showBotError('Failed to load bot status');
        });
}

// Update bot status display
function updateBotStatusDisplay(data) {
    const statusIndicator = document.getElementById('botStatusIndicator');
    const statusText = document.getElementById('botStatusText');
    const lastCheck = document.getElementById('botLastCheck');
    
    if (statusIndicator && statusText) {
        if (data.running) {
            statusIndicator.className = 'status-indicator status-online';
            statusText.textContent = 'Bot is running';
        } else {
            statusIndicator.className = 'status-indicator status-offline';
            statusText.textContent = 'Bot is stopped';
        }
    }
    
    if (lastCheck) {
        lastCheck.textContent = data.last_check || 'Never';
    }
    
    // Update bot stats
    if (data.stats) {
        const totalCommands = document.getElementById('totalCommands');
        const avgResponseTime = document.getElementById('avgResponseTime');
        
        if (totalCommands) {
            totalCommands.textContent = data.stats.total_commands || 0;
        }
        if (avgResponseTime) {
            avgResponseTime.textContent = (data.stats.avg_response_time || 0) + 'ms';
        }
    }
}

// Load bot commands
function loadBotCommands(commands) {
    const tableBody = document.getElementById('botCommandsTable');
    if (!tableBody) return;
    
    if (!commands || commands.length === 0) {
        tableBody.innerHTML = `
            <tr>
                <td colspan="4" class="text-center text-muted">
                    <i class="fas fa-info-circle me-2"></i>
                    No commands available
                </td>
            </tr>
        `;
        return;
    }
    
    tableBody.innerHTML = commands.map(cmd => `
        <tr>
            <td><code>/${cmd.command_name}</code></td>
            <td>
                <span class="badge bg-${cmd.is_enabled ? 'success' : 'danger'}">
                    ${cmd.is_enabled ? 'Enabled' : 'Disabled'}
                </span>
            </td>
            <td>${cmd.rate_limit || 30}/min</td>
            <td>
                <div class="btn-group btn-group-sm">
                    <button class="btn btn-outline-${cmd.is_enabled ? 'warning' : 'success'}" 
                            onclick="toggleBotCommand('${cmd.command_name}', ${!cmd.is_enabled})">
                        <i class="fas fa-${cmd.is_enabled ? 'pause' : 'play'}"></i>
                    </button>
                    <button class="btn btn-outline-info" onclick="editCommand('${cmd.command_name}')">
                        <i class="fas fa-edit"></i>
                    </button>
                </div>
            </td>
        </tr>
    `).join('');
}

// Load bot logs
function loadBotLogs(logs) {
    const container = document.getElementById('botLogsContainer');
    if (!container) return;
    
    if (!logs || logs.length === 0) {
        container.innerHTML = `
            <div class="text-center text-muted">
                <i class="fas fa-info-circle me-2"></i>
                No bot logs available
            </div>
        `;
        return;
    }
    
    container.innerHTML = logs.map(log => `
        <div class="log-entry">
            <span class="log-timestamp">[${new Date(log.created_at).toLocaleTimeString()}]</span>
            <span class="log-level ${log.log_type}">[${log.log_type.toUpperCase()}]</span>
            <span>${log.message}</span>
        </div>
    `).join('');
}

// Toggle bot command
function toggleBotCommand(command, enabled) {
    fetch(`?action=toggle_bot_command&command=${command}&enabled=${enabled}`)
        .then(response => response.json())
        .then(data => {
            if (data.success === false) {
                alert('Bot manager not available: ' + data.message);
                return;
            }
            
            if (data.success) {
                loadBotStatus();
            } else {
                alert('Error: ' + data.message);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error toggling command');
        });
}

// Control bot
function controlBot(action) {
    const actionText = action.charAt(0).toUpperCase() + action.slice(1);
    
    if (confirm(`Are you sure you want to ${action} the bot?`)) {
        const btn = document.getElementById(action + 'BotBtn');
        const originalText = btn.innerHTML;
        
        btn.innerHTML = '<span class="loading"></span> ' + actionText + 'ing...';
        btn.disabled = true;
        
        fetch(`?action=control_bot&bot_action=${action}`)
            .then(response => response.json())
            .then(data => {
                if (data.success === false) {
                    alert('Bot manager not available: ' + data.message);
                    return;
                }
                
                if (data.success) {
                    alert(`Bot ${action}ed successfully`);
                    loadBotStatus();
                } else {
                    alert('Error: ' + data.message);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert(`Error ${action}ing bot`);
            })
            .finally(() => {
                btn.innerHTML = originalText;
                btn.disabled = false;
            });
    }
}

// Save bot configuration
function saveBotConfig() {
    const form = document.getElementById('botConfigForm');
    const formData = new FormData(form);
    const config = {};
    
    for (let [key, value] of formData.entries()) {
        if (key === 'auto_restart' || key === 'maintenance_mode') {
            config[key] = value === 'on' ? '1' : '0';
        } else {
            config[key] = value;
        }
    }
    
    fetch('?action=update_bot_config', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify(config)
    })
    .then(response => response.json())
    .then(data => {
        if (data.success === false) {
            alert('Bot manager not available: ' + data.message);
            return;
        }
        
        if (data.success) {
            alert('Bot configuration saved successfully');
            loadBotStatus();
        } else {
            alert('Error: ' + data.message);
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Error saving configuration');
    });
}

// Refresh bot status
function refreshBotStatus() {
    loadBotStatus();
}

// Check bot connection
function checkBotConnection() {
    const btn = document.getElementById('checkConnectionBtn');
    const originalText = btn.innerHTML;
    
    // Show loading state
    btn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Checking...';
    btn.disabled = true;
    
    fetch('?action=check_bot_connection')
        .then(response => response.json())
        .then(data => {
            if (data.success === false) {
                showConnectionError(data.message);
                return;
            }
            
            // Show connection status
            showConnectionStatus(data);
            
        })
        .catch(error => {
            console.error('Error checking connection:', error);
            showConnectionError('Failed to check connection');
        })
        .finally(() => {
            // Restore button
            btn.innerHTML = originalText;
            btn.disabled = false;
        });
}

// Show connection status
function showConnectionStatus(data) {
    const statusDiv = document.getElementById('connectionStatus');
    const statusText = document.getElementById('connectionStatusText');
    const responseTime = document.getElementById('responseTime');
    const lastPing = document.getElementById('lastPing');
    
    if (statusDiv) {
        statusDiv.style.display = 'block';
        
        // Update status text with color
        if (statusText) {
            statusText.textContent = data.status;
            statusText.className = `text-${data.status_class}`;
        }
        
        // Update response time
        if (responseTime) {
            responseTime.textContent = data.response_time;
        }
        
        // Update last ping
        if (lastPing) {
            lastPing.textContent = data.timestamp;
        }
        
        // Update alert class
        const alertDiv = statusDiv.querySelector('.alert');
        if (alertDiv) {
            alertDiv.className = `alert alert-${data.status_class} mb-2`;
        }
        
        // Show bot information if available
        if (data.bot_info) {
            showBotInfo(data.bot_info);
        }
        
        // Show webhook information if available
        if (data.webhook_info) {
            showWebhookInfo(data.webhook_info);
        }
    }
}

// Show bot information
function showBotInfo(botInfo) {
    const botInfoSection = document.getElementById('botInfoSection');
    if (!botInfoSection) return;
    
    botInfoSection.style.display = 'block';
    
    // Update bot information
    const elements = {
        'botUsername': botInfo.username || '-',
        'botName': botInfo.name || '-',
        'botId': botInfo.id || '-',
        'botGroups': botInfo.can_join_groups ? 'Yes' : 'No',
        'botInline': botInfo.supports_inline_queries ? 'Yes' : 'No',
        'botReadAll': botInfo.can_read_all_group_messages ? 'Yes' : 'No'
    };
    
    Object.keys(elements).forEach(id => {
        const element = document.getElementById(id);
        if (element) {
            element.textContent = elements[id];
        }
    });
}

// Show webhook information
function showWebhookInfo(webhookInfo) {
    const webhookInfoSection = document.getElementById('webhookInfoSection');
    if (!webhookInfoSection) return;
    
    webhookInfoSection.style.display = 'block';
    
    // Update webhook information
    const elements = {
        'webhookUrl': webhookInfo.url || 'Not set',
        'webhookPending': webhookInfo.pending_update_count || '0',
        'webhookMaxConn': webhookInfo.max_connections || '-',
        'webhookCert': webhookInfo.has_custom_certificate ? 'Yes' : 'No',
        'webhookError': webhookInfo.last_error_message || 'None',
        'webhookUpdates': webhookInfo.allowed_updates ? webhookInfo.allowed_updates.join(', ') : 'All'
    };
    
    Object.keys(elements).forEach(id => {
        const element = document.getElementById(id);
        if (element) {
            element.textContent = elements[id];
        }
    });
}

// Show connection error
function showConnectionError(message) {
    const statusDiv = document.getElementById('connectionStatus');
    const statusText = document.getElementById('connectionStatusText');
    
    if (statusDiv) {
        statusDiv.style.display = 'block';
        
        if (statusText) {
            statusText.textContent = `Error: ${message}`;
            statusText.className = 'text-danger';
        }
        
        // Update alert class to danger
        const alertDiv = statusDiv.querySelector('.alert');
        if (alertDiv) {
            alertDiv.className = 'alert alert-danger mb-2';
        }
    }
}

// Edit command
function editCommand(commandName) {
    const newRateLimit = prompt(`Enter new rate limit for /${commandName}:`, '30');
    if (newRateLimit && !isNaN(newRateLimit)) {
        updateCommandRateLimit(commandName, parseInt(newRateLimit));
    }
}

// Update command rate limit
function updateCommandRateLimit(command, rateLimit) {
    fetch(`?action=update_command_rate_limit&command=${command}&rate_limit=${rateLimit}`)
        .then(response => response.json())
        .then(data => {
            if (data.success === false) {
                alert('Bot manager not available: ' + data.message);
                return;
            }
            
            if (data.success) {
                loadBotStatus();
            } else {
                alert('Error: ' + data.message);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error updating rate limit');
        });
}

// Load bot configuration
function loadBotConfig() {
    fetch('?action=get_bot_status')
        .then(response => response.json())
        .then(data => {
            if (data.success === false) {
                console.warn('Bot manager not available for config:', data.message);
                return;
            }
            
            if (data.success && data.settings) {
                const form = document.getElementById('botConfigForm');
                if (form) {
                    Object.keys(data.settings).forEach(key => {
                        const input = form.querySelector(`[name="${key}"]`);
                        if (input) {
                            if (input.type === 'checkbox') {
                                input.checked = data.settings[key].value === '1';
                            } else {
                                input.value = data.settings[key].value;
                            }
                        }
                    });
                }
            }
        })
        .catch(error => {
            console.error('Error loading bot config:', error);
        });
}

// Initialize bot management
document.addEventListener('DOMContentLoaded', function() {
    // Load initial data
    loadBotStatus();
    loadBotConfig();
    
    // Set up auto refresh for bot status
    setInterval(loadBotStatus, 30000); // Refresh every 30 seconds
});