// Broadcast JavaScript

// Load broadcasts
function loadBroadcasts() {
    fetch('?action=get_broadcasts&limit=50')
        .then(response => response.json())
        .then(data => {
            if (data.success && data.broadcasts) {
                updateBroadcastTable(data.broadcasts);
                loadBroadcastStats();
            } else {
                console.error('Error loading broadcasts:', data.message);
            }
        })
        .catch(error => console.error('Error:', error));
}

// Update broadcast table
function updateBroadcastTable(broadcasts) {
    const tableBody = document.getElementById('broadcastTable');
    if (!tableBody) return;
    
    if (broadcasts.length === 0) {
        tableBody.innerHTML = '<tr><td colspan="7" class="text-center text-muted">No broadcasts found</td></tr>';
        return;
    }
    
    tableBody.innerHTML = broadcasts.map(broadcast => `
        <tr>
            <td>${broadcast.id}</td>
            <td>${broadcast.message.length > 50 ? broadcast.message.substring(0, 50) + '...' : broadcast.message}</td>
            <td>
                <span class="badge bg-${getStatusColor(broadcast.status)}">
                    ${broadcast.status}
                </span>
            </td>
            <td>${broadcast.success_count || 0}</td>
            <td>${broadcast.failed_count || 0}</td>
            <td>${broadcast.created_at ? new Date(broadcast.created_at).toLocaleString() : 'N/A'}</td>
            <td>
                <div class="btn-group btn-group-sm">
                    <button class="btn btn-outline-info" onclick="viewBroadcast(${broadcast.id})">
                        <i class="fas fa-eye"></i>
                    </button>
                    ${broadcast.status === 'pending' || broadcast.status === 'running' ? 
                        `<button class="btn btn-outline-danger" onclick="cancelBroadcast(${broadcast.id})">
                            <i class="fas fa-times"></i>
                        </button>` : ''
                    }
                </div>
            </td>
        </tr>
    `).join('');
}

// Get status color
function getStatusColor(status) {
    switch (status) {
        case 'completed': return 'success';
        case 'running': return 'warning';
        case 'failed': return 'danger';
        case 'cancelled': return 'secondary';
        default: return 'info';
    }
}

// Load broadcast statistics
function loadBroadcastStats() {
    fetch('?action=get_broadcast_stats')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                updateBroadcastStats(data);
            }
        })
        .catch(error => console.error('Error loading broadcast stats:', error));
}

// Update broadcast statistics
function updateBroadcastStats(stats) {
    const totalBroadcasts = document.getElementById('totalBroadcasts');
    const successfulBroadcasts = document.getElementById('successfulBroadcasts');
    const totalMessagesSent = document.getElementById('totalMessagesSent');
    const recentBroadcasts = document.getElementById('recentBroadcasts');
    
    if (totalBroadcasts) totalBroadcasts.textContent = stats.total_broadcasts || 0;
    if (successfulBroadcasts) successfulBroadcasts.textContent = stats.successful_broadcasts || 0;
    if (totalMessagesSent) totalMessagesSent.textContent = stats.total_messages_sent || 0;
    if (recentBroadcasts) recentBroadcasts.textContent = stats.recent_broadcasts || 0;
}

// Show broadcast modal
function showBroadcastModal() {
    const modal = new bootstrap.Modal(document.getElementById('newBroadcastModal'));
    modal.show();
}

// Send broadcast
function sendBroadcast() {
    const form = document.getElementById('broadcastForm');
    const formData = new FormData(form);
    const data = {};
    
    for (let [key, value] of formData.entries()) {
        if (key === 'target_users' && value.trim() === '') {
            continue; // Skip empty target_users
        }
        data[key] = value;
    }
    
    // Validate message
    if (!data.message || data.message.trim() === '') {
        alert('Please enter a message');
        return;
    }
    
    // Validate message length
    if (data.message.length > 4000) {
        alert('Message is too long. Maximum 4000 characters.');
        return;
    }
    
    // Show loading
    const sendBtn = document.querySelector('#newBroadcastModal .btn-primary');
    const originalText = sendBtn.innerHTML;
    sendBtn.innerHTML = '<span class="loading"></span> Sending...';
    sendBtn.disabled = true;
    
    fetch('?action=create_broadcast', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify(data)
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            alert('Broadcast created successfully');
            bootstrap.Modal.getInstance(document.getElementById('newBroadcastModal')).hide();
            loadBroadcasts();
            form.reset();
        } else {
            alert('Error: ' + data.message);
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Error creating broadcast');
    })
    .finally(() => {
        sendBtn.innerHTML = originalText;
        sendBtn.disabled = false;
    });
}

// View broadcast details
function viewBroadcast(broadcastId) {
    fetch(`?action=get_broadcast_details&id=${broadcastId}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showBroadcastDetails(data.broadcast);
            } else {
                alert('Error: ' + data.message);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error loading broadcast details');
        });
}

// Show broadcast details
function showBroadcastDetails(broadcast) {
    const modal = new bootstrap.Modal(document.getElementById('broadcastPreviewModal'));
    const content = document.getElementById('broadcastPreviewContent');
    
    content.innerHTML = `
        <div class="mb-3">
            <strong>ID:</strong> ${broadcast.id}
        </div>
        <div class="mb-3">
            <strong>Status:</strong> 
            <span class="badge bg-${getStatusColor(broadcast.status)}">${broadcast.status}</span>
        </div>
        <div class="mb-3">
            <strong>Message:</strong>
            <div class="border rounded p-2 mt-1">${broadcast.message}</div>
        </div>
        <div class="row mb-3">
            <div class="col-6">
                <strong>Success:</strong> ${broadcast.success_count || 0}
            </div>
            <div class="col-6">
                <strong>Failed:</strong> ${broadcast.failed_count || 0}
            </div>
        </div>
        <div class="mb-3">
            <strong>Created:</strong> ${broadcast.created_at ? new Date(broadcast.created_at).toLocaleString() : 'N/A'}
        </div>
        ${broadcast.completed_at ? `
        <div class="mb-3">
            <strong>Completed:</strong> ${new Date(broadcast.completed_at).toLocaleString()}
        </div>
        ` : ''}
    `;
    
    modal.show();
}

// Cancel broadcast
function cancelBroadcast(broadcastId) {
    if (confirm('Are you sure you want to cancel this broadcast?')) {
        fetch(`?action=cancel_broadcast&id=${broadcastId}`)
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert('Broadcast cancelled successfully');
                    loadBroadcasts();
                } else {
                    alert('Error: ' + data.message);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Error cancelling broadcast');
            });
    }
}

// Preview broadcast message
function previewBroadcast() {
    const form = document.getElementById('broadcastForm');
    const message = form.querySelector('[name="message"]').value;
    
    if (!message.trim()) {
        alert('Please enter a message to preview');
        return;
    }
    
    const modal = new bootstrap.Modal(document.getElementById('broadcastPreviewModal'));
    const content = document.getElementById('broadcastPreviewContent');
    
    content.innerHTML = `
        <div class="mb-3">
            <strong>Preview:</strong>
        </div>
        <div class="border rounded p-3 bg-light">
            ${message.replace(/\n/g, '<br>')}
        </div>
    `;
    
    modal.show();
}

// Handle target type change
document.addEventListener('DOMContentLoaded', function() {
    const targetTypeSelect = document.getElementById('targetType');
    const specificUsersDiv = document.getElementById('specificUsersDiv');
    
    if (targetTypeSelect && specificUsersDiv) {
        targetTypeSelect.addEventListener('change', function() {
            if (this.value === 'specific') {
                specificUsersDiv.style.display = 'block';
            } else {
                specificUsersDiv.style.display = 'none';
            }
        });
    }
    
    // Add preview button
    const messageTextarea = document.querySelector('[name="message"]');
    if (messageTextarea) {
        const previewBtn = document.createElement('button');
        previewBtn.type = 'button';
        previewBtn.className = 'btn btn-outline-secondary btn-sm mt-2';
        previewBtn.innerHTML = '<i class="fas fa-eye me-2"></i>Preview';
        previewBtn.onclick = previewBroadcast;
        
        messageTextarea.parentNode.appendChild(previewBtn);
    }
});

// Initialize broadcast management
document.addEventListener('DOMContentLoaded', function() {
    // Load initial data
    loadBroadcasts();
    
    // Set up auto refresh
    setInterval(loadBroadcasts, 60000); // Refresh every minute
});