-- Admin Panel Database Schema
-- Run this SQL to create the necessary tables for the admin panel

-- Admin users table
CREATE TABLE IF NOT EXISTS `admin_users` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `username` varchar(50) NOT NULL UNIQUE,
    `password` varchar(255) NOT NULL,
    `role` enum('super_admin', 'admin', 'moderator') DEFAULT 'admin',
    `is_active` tinyint(1) DEFAULT 1,
    `last_login` datetime NULL,
    `failed_attempts` int(11) DEFAULT 0,
    `last_failed_attempt` datetime NULL,
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Insert default admin user (password: admin123)
INSERT INTO `admin_users` (`username`, `password`, `role`) VALUES 
('admin', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'super_admin')
ON DUPLICATE KEY UPDATE `username` = `username`;

-- Bot settings table
CREATE TABLE IF NOT EXISTS `bot_settings` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `setting_name` varchar(100) NOT NULL UNIQUE,
    `setting_value` text,
    `description` text,
    `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Insert default bot settings
INSERT INTO `bot_settings` (`setting_name`, `setting_value`, `description`) VALUES
('max_connections', '100', 'Maximum webhook connections'),
('rate_limit', '30', 'Rate limit per minute per user'),
('response_delay', '1', 'Auto response delay in seconds'),
('log_level', 'INFO', 'Bot logging level'),
('auto_restart', '1', 'Auto restart on failure'),
('maintenance_mode', '0', 'Maintenance mode flag')
ON DUPLICATE KEY UPDATE `setting_name` = `setting_name`;

-- Bot commands table
CREATE TABLE IF NOT EXISTS `bot_commands` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `command_name` varchar(50) NOT NULL UNIQUE,
    `is_enabled` tinyint(1) DEFAULT 1,
    `rate_limit` int(11) DEFAULT 30,
    `description` text,
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Insert default bot commands
INSERT INTO `bot_commands` (`command_name`, `is_enabled`, `rate_limit`, `description`) VALUES
('start', 1, 10, 'Start the bot and show welcome message'),
('help', 1, 10, 'Show help information'),
('cari', 1, 30, 'Search for movies'),
('adult', 1, 20, 'Search for adult content'),
('adult_random', 1, 10, 'Get random adult content'),
('broadcast', 1, 5, 'Admin broadcast command'),
('stats', 1, 5, 'Show bot statistics')
ON DUPLICATE KEY UPDATE `command_name` = `command_name`;

-- Broadcast templates table
CREATE TABLE IF NOT EXISTS `broadcast_templates` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `name` varchar(100) NOT NULL,
    `content` text NOT NULL,
    `variables` json,
    `is_active` tinyint(1) DEFAULT 1,
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Insert default broadcast templates
INSERT INTO `broadcast_templates` (`name`, `content`, `variables`) VALUES
('welcome', '🎉 Welcome to our bot!\n\nUse /cari to search for movies or /adult for adult content.', '[]'),
('maintenance', '🔧 Bot is currently under maintenance. We will be back soon!', '[]'),
('update', '📢 Bot has been updated with new features! Check /help for more info.', '[]')
ON DUPLICATE KEY UPDATE `name` = `name`;

-- Broadcast recipients table
CREATE TABLE IF NOT EXISTS `broadcast_recipients` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `broadcast_id` int(11) NOT NULL,
    `user_id` bigint(20) NOT NULL,
    `status` enum('pending', 'delivered', 'failed') DEFAULT 'pending',
    `error_message` text,
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    KEY `idx_broadcast_id` (`broadcast_id`),
    KEY `idx_user_id` (`user_id`),
    KEY `idx_status` (`status`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- System logs table (if not exists)
CREATE TABLE IF NOT EXISTS `system_logs` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `level` enum('DEBUG', 'INFO', 'WARNING', 'ERROR', 'CRITICAL') NOT NULL,
    `message` text NOT NULL,
    `context` json,
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    KEY `idx_level` (`level`),
    KEY `idx_created_at` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Log settings table (if not exists)
CREATE TABLE IF NOT EXISTS `log_settings` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `setting_name` varchar(100) NOT NULL UNIQUE,
    `setting_value` varchar(255),
    `description` text,
    `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Insert default log settings
INSERT INTO `log_settings` (`setting_name`, `setting_value`, `description`) VALUES
('auto_reset_hours', '24', 'Auto reset logs every X hours'),
('max_log_entries', '10000', 'Maximum log entries to keep'),
('log_retention_days', '30', 'Log retention period in days'),
('log_level', 'INFO', 'Minimum log level to store')
ON DUPLICATE KEY UPDATE `setting_name` = `setting_name`;

-- Bot logs table (if not exists)
CREATE TABLE IF NOT EXISTS `bot_logs` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `log_type` varchar(50) NOT NULL,
    `message` text NOT NULL,
    `user_id` bigint(20),
    `username` varchar(100),
    `action` varchar(100),
    `details` text,
    `response_time` decimal(10,3),
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    KEY `idx_log_type` (`log_type`),
    KEY `idx_user_id` (`user_id`),
    KEY `idx_created_at` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- User searches table (if not exists)
CREATE TABLE IF NOT EXISTS `user_searches` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `user_id` bigint(20) NOT NULL,
    `username` varchar(100),
    `search_query` varchar(255) NOT NULL,
    `search_type` enum('movie', 'adult') NOT NULL,
    `results_count` int(11) DEFAULT 0,
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    KEY `idx_user_id` (`user_id`),
    KEY `idx_search_type` (`search_type`),
    KEY `idx_created_at` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Users table (if not exists)
CREATE TABLE IF NOT EXISTS `users` (
    `id` bigint(20) NOT NULL,
    `username` varchar(100),
    `first_name` varchar(100),
    `last_name` varchar(100),
    `last_seen` timestamp NULL,
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    KEY `idx_username` (`username`),
    KEY `idx_last_seen` (`last_seen`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Broadcast logs table (if not exists)
CREATE TABLE IF NOT EXISTS `broadcast_logs` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `admin_id` int(11) NOT NULL,
    `message` text NOT NULL,
    `status` enum('pending', 'running', 'completed', 'cancelled', 'failed') DEFAULT 'pending',
    `target_type` enum('all', 'active', 'specific') DEFAULT 'all',
    `target_users` json,
    `scheduled_at` datetime NULL,
    `started_at` datetime NULL,
    `completed_at` datetime NULL,
    `cancelled_at` datetime NULL,
    `cancelled_by` int(11) NULL,
    `template_id` int(11) NULL,
    `priority` enum('low', 'normal', 'high') DEFAULT 'normal',
    `success_count` int(11) DEFAULT 0,
    `failed_count` int(11) DEFAULT 0,
    `total_users` int(11) DEFAULT 0,
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    KEY `idx_admin_id` (`admin_id`),
    KEY `idx_status` (`status`),
    KEY `idx_created_at` (`created_at`),
    FOREIGN KEY (`admin_id`) REFERENCES `admin_users`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`template_id`) REFERENCES `broadcast_templates`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
