-- Fix Admin Schema - Safe Update Script
-- This script will safely update existing tables or create new ones

-- Check if admin_users table exists and has correct structure
-- If not, create it or alter it
CREATE TABLE IF NOT EXISTS `admin_users` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `username` varchar(50) NOT NULL UNIQUE,
    `password` varchar(255) NOT NULL,
    `role` enum('super_admin', 'admin', 'moderator') DEFAULT 'admin',
    `is_active` tinyint(1) DEFAULT 1,
    `last_login` datetime NULL,
    `failed_attempts` int(11) DEFAULT 0,
    `last_failed_attempt` datetime NULL,
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Add missing columns if they don't exist
ALTER TABLE `admin_users` 
ADD COLUMN IF NOT EXISTS `password` varchar(255) NOT NULL AFTER `username`,
ADD COLUMN IF NOT EXISTS `role` enum('super_admin', 'admin', 'moderator') DEFAULT 'admin' AFTER `password`,
ADD COLUMN IF NOT EXISTS `is_active` tinyint(1) DEFAULT 1 AFTER `role`,
ADD COLUMN IF NOT EXISTS `last_login` datetime NULL AFTER `is_active`,
ADD COLUMN IF NOT EXISTS `failed_attempts` int(11) DEFAULT 0 AFTER `last_login`,
ADD COLUMN IF NOT EXISTS `last_failed_attempt` datetime NULL AFTER `failed_attempts`,
ADD COLUMN IF NOT EXISTS `created_at` timestamp DEFAULT CURRENT_TIMESTAMP AFTER `last_failed_attempt`,
ADD COLUMN IF NOT EXISTS `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP AFTER `created_at`;

-- Insert default admin user only if it doesn't exist
INSERT IGNORE INTO `admin_users` (`username`, `password`, `role`) VALUES 
('admin', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'super_admin');

-- Create bot_settings table
CREATE TABLE IF NOT EXISTS `bot_settings` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `setting_name` varchar(100) NOT NULL UNIQUE,
    `setting_value` text,
    `description` text,
    `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Insert default bot settings
INSERT IGNORE INTO `bot_settings` (`setting_name`, `setting_value`, `description`) VALUES
('max_connections', '100', 'Maximum webhook connections'),
('rate_limit', '30', 'Rate limit per minute per user'),
('response_delay', '1', 'Auto response delay in seconds'),
('log_level', 'INFO', 'Bot logging level'),
('auto_restart', '1', 'Auto restart on failure'),
('maintenance_mode', '0', 'Maintenance mode flag');

-- Create bot_commands table
CREATE TABLE IF NOT EXISTS `bot_commands` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `command_name` varchar(50) NOT NULL UNIQUE,
    `is_enabled` tinyint(1) DEFAULT 1,
    `rate_limit` int(11) DEFAULT 30,
    `description` text,
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Insert default bot commands
INSERT IGNORE INTO `bot_commands` (`command_name`, `is_enabled`, `rate_limit`, `description`) VALUES
('start', 1, 10, 'Start the bot and show welcome message'),
('help', 1, 10, 'Show help information'),
('cari', 1, 30, 'Search for movies'),
('adult', 1, 20, 'Search for adult content'),
('adult_random', 1, 10, 'Get random adult content'),
('broadcast', 1, 5, 'Admin broadcast command'),
('stats', 1, 5, 'Show bot statistics');

-- Create broadcast_templates table
CREATE TABLE IF NOT EXISTS `broadcast_templates` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `name` varchar(100) NOT NULL,
    `content` text NOT NULL,
    `variables` json,
    `is_active` tinyint(1) DEFAULT 1,
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Insert default broadcast templates
INSERT IGNORE INTO `broadcast_templates` (`name`, `content`, `variables`) VALUES
('welcome', '🎉 Welcome to our bot!\n\nUse /cari to search for movies or /adult for adult content.', '[]'),
('maintenance', '🔧 Bot is currently under maintenance. We will be back soon!', '[]'),
('update', '📢 Bot has been updated with new features! Check /help for more info.', '[]');

-- Create broadcast_recipients table
CREATE TABLE IF NOT EXISTS `broadcast_recipients` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `broadcast_id` int(11) NOT NULL,
    `user_id` bigint(20) NOT NULL,
    `status` enum('pending', 'delivered', 'failed') DEFAULT 'pending',
    `error_message` text,
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    KEY `idx_broadcast_id` (`broadcast_id`),
    KEY `idx_user_id` (`user_id`),
    KEY `idx_status` (`status`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Create system_logs table
CREATE TABLE IF NOT EXISTS `system_logs` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `level` enum('DEBUG', 'INFO', 'WARNING', 'ERROR', 'CRITICAL') NOT NULL,
    `message` text NOT NULL,
    `context` json,
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    KEY `idx_level` (`level`),
    KEY `idx_created_at` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Create log_settings table
CREATE TABLE IF NOT EXISTS `log_settings` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `setting_name` varchar(100) NOT NULL UNIQUE,
    `setting_value` varchar(255),
    `description` text,
    `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Insert default log settings
INSERT IGNORE INTO `log_settings` (`setting_name`, `setting_value`, `description`) VALUES
('auto_reset_hours', '24', 'Auto reset logs every X hours'),
('max_log_entries', '10000', 'Maximum log entries to keep'),
('log_retention_days', '30', 'Log retention period in days'),
('log_level', 'INFO', 'Minimum log level to store');

-- Create broadcast_logs table
CREATE TABLE IF NOT EXISTS `broadcast_logs` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `admin_id` int(11) NOT NULL,
    `message` text NOT NULL,
    `status` enum('pending', 'running', 'completed', 'cancelled', 'failed') DEFAULT 'pending',
    `target_type` enum('all', 'active', 'specific') DEFAULT 'all',
    `target_users` json,
    `scheduled_at` datetime NULL,
    `started_at` datetime NULL,
    `completed_at` datetime NULL,
    `cancelled_at` datetime NULL,
    `cancelled_by` int(11) NULL,
    `template_id` int(11) NULL,
    `priority` enum('low', 'normal', 'high') DEFAULT 'normal',
    `success_count` int(11) DEFAULT 0,
    `failed_count` int(11) DEFAULT 0,
    `total_users` int(11) DEFAULT 0,
    `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
    `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    KEY `idx_admin_id` (`admin_id`),
    KEY `idx_status` (`status`),
    KEY `idx_created_at` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Add foreign key constraints if they don't exist
-- Note: These might fail if the referenced tables don't exist yet
-- That's okay, we'll add them after all tables are created

-- Show completion message
SELECT 'Admin schema setup completed successfully!' as message;
