<?php
/**
 * Migrate Configuration to Database
 * Run this ONCE to migrate all configuration from .env to database
 */

require_once '../../config/database.php';

echo "<h2>Migrate Configuration to Database</h2>\n";

try {
    $database = new Database();
    $pdo = $database->getConnection();
    
    // Configuration from .env file
    $configurations = [
        // Bot Configuration
        ['bot_token', '7116738691:AAFw-vLy8xYoDNg5urgHrj_pyVKKHT-vw8w', 'Telegram Bot Token'],
        ['webhook_url', 'https://bot.apivalidasi.my.id/telegram_bot.php', 'Webhook URL'],
        
        // Domain Configuration
        ['domain', 'https://bot.apivalidasi.my.id', 'Main Domain'],
        ['public_domain', 'https://bot.apivalidasi.my.id/public', 'Public Domain'],
        
        // API Configuration
        ['adult_api_url', 'https://tes.apivalidasi.my.id/api_adult.php', 'Adult API URL'],
        
        // Rate Limiting
        ['rate_limit_message', '30', 'Rate limit per message'],
        ['rate_limit_search', '10', 'Rate limit per search'],
        ['rate_limit_adult', '5', 'Rate limit per adult'],
        ['rate_limit_window', '1', 'Rate limit window in minutes'],
        
        // Cache Configuration
        ['cache_timeout_movies', '3600', 'Cache timeout for movies'],
        ['cache_timeout_adult', '1800', 'Cache timeout for adult'],
        ['cache_timeout_details', '3600', 'Cache timeout for details'],
        
        // Security
        ['admin_ids', '6051115565', 'Admin User IDs'],
        ['enable_rate_limiting', 'true', 'Enable rate limiting'],
        ['enable_input_sanitization', 'true', 'Enable input sanitization'],
        
        // Logging
        ['log_level', 'INFO', 'Log level'],
        ['log_file', 'logs/bot.log', 'Log file path'],
        ['max_log_size', '10485760', 'Max log size in bytes'],
        ['max_log_files', '5', 'Max log files'],
    ];
    
    // Check if bot_settings table exists
    $stmt = $pdo->query("SHOW TABLES LIKE 'bot_settings'");
    $tableExists = $stmt->rowCount() > 0;
    
    if (!$tableExists) {
        echo "Creating bot_settings table...<br>\n";
        
        $sql = "CREATE TABLE `bot_settings` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `setting_name` varchar(100) NOT NULL UNIQUE,
            `setting_value` text,
            `description` text,
            `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";
        
        $pdo->exec($sql);
        echo "✅ bot_settings table created<br>\n";
    }
    
    // Check if app_settings table exists
    $stmt = $pdo->query("SHOW TABLES LIKE 'app_settings'");
    $tableExists = $stmt->rowCount() > 0;
    
    if (!$tableExists) {
        echo "Creating app_settings table...<br>\n";
        
        $sql = "CREATE TABLE `app_settings` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `setting_name` varchar(100) NOT NULL UNIQUE,
            `setting_value` text,
            `description` text,
            `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";
        
        $pdo->exec($sql);
        echo "✅ app_settings table created<br>\n";
    }
    
    // Insert configurations
    echo "<br><h3>Inserting configurations:</h3>\n";
    
    $inserted = 0;
    $updated = 0;
    
    foreach ($configurations as $config) {
        $stmt = $pdo->prepare("
            INSERT INTO bot_settings (setting_name, setting_value, description) 
            VALUES (?, ?, ?) 
            ON DUPLICATE KEY UPDATE 
            setting_value = VALUES(setting_value),
            description = VALUES(description)
        ");
        
        if ($stmt->execute($config)) {
            $existing = $pdo->query("SELECT COUNT(*) FROM bot_settings WHERE setting_name = '{$config[0]}' AND setting_value = '{$config[1]}'")->fetchColumn();
            if ($existing) {
                $updated++;
            } else {
                $inserted++;
            }
            echo "✅ {$config[0]}: {$config[1]}<br>\n";
        }
    }
    
    echo "<br><h3>Migration Summary:</h3>\n";
    echo "Inserted: $inserted configurations<br>\n";
    echo "Updated: $updated configurations<br>\n";
    
    // Verify configurations
    echo "<br><h3>Verifying configurations:</h3>\n";
    $stmt = $pdo->query("SELECT COUNT(*) FROM bot_settings");
    $count = $stmt->fetchColumn();
    echo "Total configurations in database: $count<br>\n";
    
    // Show all configurations
    echo "<br><h3>All configurations:</h3>\n";
    $stmt = $pdo->query("SELECT setting_name, setting_value FROM bot_settings ORDER BY setting_name");
    echo "<table border='1' style='border-collapse: collapse; width: 100%;'>\n";
    echo "<tr><th>Setting Name</th><th>Value</th></tr>\n";
    while ($row = $stmt->fetch()) {
        $value = strlen($row['setting_value']) > 50 ? substr($row['setting_value'], 0, 50) . '...' : $row['setting_value'];
        echo "<tr><td>{$row['setting_name']}</td><td>" . htmlspecialchars($value) . "</td></tr>\n";
    }
    echo "</table>\n";
    
    echo "<br><h3>✅ Migration completed successfully!</h3>\n";
    echo "All configurations are now stored in the database.<br>\n";
    echo "You can now remove the .env file and use database for all configurations.<br>\n";
    
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "<br>\n";
    echo "Stack trace:<br>\n";
    echo "<pre>" . $e->getTraceAsString() . "</pre>\n";
}
?>
