<?php
/**
 * Admin Controller
 * Handles all admin panel requests and data processing
 */

class AdminController {
    private $pdo;
    private $logger;
    private $botManager;
    private $broadcastManager;
    
    public function __construct(PDO $pdo, DatabaseLogger $logger) {
        $this->pdo = $pdo;
        $this->logger = $logger;
        
        // Initialize managers only if classes exist
        try {
            if (class_exists('BotManager')) {
                $this->botManager = new BotManager($pdo, $logger);
            } else {
                $this->botManager = null;
            }
        } catch (Exception $e) {
            $this->botManager = null;
        }
        
        try {
            if (class_exists('BroadcastManager')) {
                $this->broadcastManager = new BroadcastManager($pdo, $logger);
            } else {
                $this->broadcastManager = null;
            }
        } catch (Exception $e) {
            $this->broadcastManager = null;
        }
    }
    
    /**
     * Handle AJAX requests
     */
    public function handleAjaxRequest(string $action): void {
        header('Content-Type: application/json');
        
        try {
            switch ($action) {
                case 'get_stats':
                    echo json_encode($this->getStats());
                    break;
                    
                case 'get_logs':
                    $type = $_GET['type'] ?? 'bot';
                    $limit = (int)($_GET['limit'] ?? 50);
                    echo json_encode($this->getLogs($type, $limit));
                    break;
                    
                case 'clear_logs':
                    echo json_encode($this->clearLogs());
                    break;
                    
                case 'update_settings':
                    $input = json_decode(file_get_contents('php://input'), true);
                    echo json_encode($this->updateSettings($input));
                    break;
                    
                case 'get_users':
                    $limit = (int)($_GET['limit'] ?? 50);
                    $offset = (int)($_GET['offset'] ?? 0);
                    echo json_encode($this->getUsers($limit, $offset));
                    break;
                    
                case 'get_broadcasts':
                    $limit = (int)($_GET['limit'] ?? 50);
                    $offset = (int)($_GET['offset'] ?? 0);
                    echo json_encode($this->getBroadcasts($limit, $offset));
                    break;
                    
                case 'create_broadcast':
                    $input = json_decode(file_get_contents('php://input'), true);
                    echo json_encode($this->createBroadcast($input));
                    break;
                    
                case 'get_bot_status':
                    echo json_encode($this->getBotStatus());
                    break;
                    
                case 'update_bot_config':
                    $input = json_decode(file_get_contents('php://input'), true);
                    echo json_encode($this->updateBotConfig($input));
                    break;
                    
                case 'toggle_bot_command':
                    $command = $_GET['command'] ?? '';
                    $enabled = $_GET['enabled'] === 'true';
                    echo json_encode($this->toggleBotCommand($command, $enabled));
                    break;
                    
                case 'update_command_rate_limit':
                    $command = $_GET['command'] ?? '';
                    $rateLimit = (int)($_GET['rate_limit'] ?? 30);
                    
                    // Debug logging
                    error_log("Update command rate limit: command=$command, rateLimit=$rateLimit");
                    
                    echo json_encode($this->updateCommandRateLimit($command, $rateLimit));
                    break;
                    
                case 'control_bot':
                    $action = $_GET['bot_action'] ?? '';
                    echo json_encode($this->controlBot($action));
                    break;
                    
                case 'check_bot_connection':
                    echo json_encode($this->checkBotConnection());
                    break;
                    
                default:
                    echo json_encode(['success' => false, 'message' => 'Invalid action']);
                    break;
            }
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        }
    }
    
    /**
     * Get system statistics
     */
    public function getStats(): array {
        try {
            // Get user count
            $stmt = $this->pdo->query("SELECT COUNT(*) as total_users FROM users");
            $totalUsers = $stmt->fetchColumn();
            
            // Get search count
            $stmt = $this->pdo->query("SELECT COUNT(*) as total_searches FROM user_searches");
            $totalSearches = $stmt->fetchColumn();
            
            // Get ad clicks
            $stmt = $this->pdo->query("SELECT COUNT(*) as total_ad_clicks FROM bot_logs WHERE log_type = 'ad_click'");
            $totalAdClicks = $stmt->fetchColumn();
            
            // Get recent activity
            $stmt = $this->pdo->query("SELECT COUNT(*) as recent_activity FROM user_searches WHERE created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)");
            $recentActivity = $stmt->fetchColumn();
            
            return [
                'total_users' => $totalUsers,
                'total_searches' => $totalSearches,
                'total_ad_clicks' => $totalAdClicks,
                'recent_activity' => $recentActivity
            ];
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Failed to get stats: ' . $e->getMessage());
            return [
                'total_users' => 0,
                'total_searches' => 0,
                'total_ad_clicks' => 0,
                'recent_activity' => 0
            ];
        }
    }
    
    /**
     * Get logs
     */
    public function getLogs(string $type, int $limit): array {
        try {
            if ($type === 'bot') {
                $stmt = $this->pdo->prepare("
                    SELECT * FROM bot_logs 
                    ORDER BY created_at DESC 
                    LIMIT ?
                ");
            } else {
                $stmt = $this->pdo->prepare("
                    SELECT * FROM system_logs 
                    ORDER BY created_at DESC 
                    LIMIT ?
                ");
            }
            $stmt->execute([$limit]);
            return $stmt->fetchAll();
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Failed to get logs: ' . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Clear logs
     */
    public function clearLogs(): array {
        try {
            $stmt = $this->pdo->prepare("DELETE FROM bot_logs");
            $stmt->execute();
            
            $this->logger->log('INFO', 'Bot logs cleared by admin');
            
            return ['success' => true, 'message' => 'Logs cleared successfully'];
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Failed to clear logs: ' . $e->getMessage());
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }
    
    /**
     * Update settings
     */
    public function updateSettings(array $settings): array {
        try {
            foreach ($settings as $key => $value) {
                $stmt = $this->pdo->prepare("
                    INSERT INTO log_settings (setting_name, setting_value) 
                    VALUES (?, ?) 
                    ON DUPLICATE KEY UPDATE setting_value = ?
                ");
                $stmt->execute([$key, $value, $value]);
            }
            
            $this->logger->log('INFO', 'Settings updated by admin');
            
            return ['success' => true, 'message' => 'Settings updated successfully'];
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Failed to update settings: ' . $e->getMessage());
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }
    
    /**
     * Get users
     */
    public function getUsers(int $limit, int $offset): array {
        try {
            $stmt = $this->pdo->prepare("
                SELECT id, username, first_name, last_name, last_seen, created_at 
                FROM users 
                ORDER BY last_seen DESC 
                LIMIT ? OFFSET ?
            ");
            $stmt->execute([$limit, $offset]);
            $users = $stmt->fetchAll();
            
            // Get total count
            $stmt = $this->pdo->query("SELECT COUNT(*) as total FROM users");
            $total = $stmt->fetchColumn();
            
            return [
                'success' => true,
                'users' => $users,
                'total' => $total,
                'limit' => $limit,
                'offset' => $offset
            ];
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Failed to get users: ' . $e->getMessage());
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }
    
    /**
     * Get broadcasts
     */
    public function getBroadcasts(int $limit, int $offset): array {
        try {
            $stmt = $this->pdo->prepare("
                SELECT id, message, status, success_count, failed_count, total_users, 
                       created_at, completed_at 
                FROM broadcast_logs 
                ORDER BY created_at DESC 
                LIMIT ? OFFSET ?
            ");
            $stmt->execute([$limit, $offset]);
            $broadcasts = $stmt->fetchAll();
            
            return [
                'success' => true,
                'broadcasts' => $broadcasts
            ];
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Failed to get broadcasts: ' . $e->getMessage());
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }
    
    /**
     * Create broadcast
     */
    public function createBroadcast(array $data): array {
        if (!$this->broadcastManager) {
            return ['success' => false, 'message' => 'Broadcast manager not available'];
        }
        
        try {
            $message = $data['message'] ?? '';
            $scheduleTime = $data['schedule_time'] ?? null;
            $targetUsers = $data['target_users'] ?? null;
            
            if (empty($message)) {
                return ['success' => false, 'message' => 'Message is required'];
            }
            
            $broadcastId = $this->broadcastManager->createBroadcast($message, $scheduleTime, $targetUsers);
            
            if ($broadcastId) {
                $this->logger->log('INFO', 'Broadcast created by admin: ' . $broadcastId);
                return ['success' => true, 'message' => 'Broadcast created successfully', 'id' => $broadcastId];
            } else {
                return ['success' => false, 'message' => 'Failed to create broadcast'];
            }
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Failed to create broadcast: ' . $e->getMessage());
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }
    
    /**
     * Get bot status
     */
    public function getBotStatus(): array {
        if (!$this->botManager) {
            return [
                'success' => false,
                'message' => 'Bot manager not available',
                'running' => false,
                'settings' => [],
                'commands' => [],
                'recent_logs' => []
            ];
        }
        
        try {
            return $this->botManager->getBotStatus();
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Failed to get bot status: ' . $e->getMessage());
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }
    
    /**
     * Update bot configuration
     */
    public function updateBotConfig(array $config): array {
        if (!$this->botManager) {
            return ['success' => false, 'message' => 'Bot manager not available'];
        }
        
        try {
            $success = $this->botManager->updateBotConfig($config);
            
            if ($success) {
                $this->logger->log('INFO', 'Bot configuration updated by admin');
                return ['success' => true, 'message' => 'Bot configuration updated successfully'];
            } else {
                return ['success' => false, 'message' => 'Failed to update bot configuration'];
            }
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Failed to update bot config: ' . $e->getMessage());
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }
    
    /**
     * Toggle bot command
     */
    public function toggleBotCommand(string $command, bool $enabled): array {
        if (!$this->botManager) {
            return ['success' => false, 'message' => 'Bot manager not available'];
        }
        
        try {
            $success = $this->botManager->toggleBotCommand($command, $enabled);
            
            if ($success) {
                $this->logger->log('INFO', "Bot command '$command' " . ($enabled ? 'enabled' : 'disabled') . ' by admin');
                return ['success' => true, 'message' => 'Command status updated successfully'];
            } else {
                return ['success' => false, 'message' => 'Failed to update command status'];
            }
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Failed to toggle bot command: ' . $e->getMessage());
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }
    
    /**
     * Control bot
     */
    public function controlBot(string $action): array {
        if (!$this->botManager) {
            return ['success' => false, 'message' => 'Bot manager not available'];
        }
        
        try {
            $success = false;
            
            switch ($action) {
                case 'start':
                    $success = $this->botManager->startBot();
                    break;
                case 'stop':
                    $success = $this->botManager->stopBot();
                    break;
                case 'restart':
                    $success = $this->botManager->restartBot();
                    break;
                default:
                    return ['success' => false, 'message' => 'Invalid bot action'];
            }
            
            if ($success) {
                $this->logger->log('INFO', "Bot $action executed by admin");
                return ['success' => true, 'message' => "Bot $action executed successfully"];
            } else {
                return ['success' => false, 'message' => "Failed to $action bot"];
            }
        } catch (Exception $e) {
            $this->logger->log('ERROR', "Failed to control bot: " . $e->getMessage());
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }
    
    /**
     * Update command rate limit
     */
    public function updateCommandRateLimit(string $command, int $rateLimit): array {
        if (!$this->botManager) {
            return ['success' => false, 'message' => 'Bot manager not available'];
        }
        
        try {
            $success = $this->botManager->updateCommandRateLimit($command, $rateLimit);
            
            if ($success) {
                $this->logger->log('INFO', "Updated rate limit for command '$command' to $rateLimit");
                return ['success' => true, 'message' => 'Rate limit updated successfully'];
            } else {
                return ['success' => false, 'message' => 'Failed to update rate limit'];
            }
        } catch (Exception $e) {
            $this->logger->log('ERROR', "Failed to update command rate limit: " . $e->getMessage());
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }
    
    /**
     * Check bot connection
     */
    public function checkBotConnection(): array {
        if (!$this->botManager) {
            return ['success' => false, 'message' => 'Bot manager not available'];
        }
        
        try {
            $startTime = microtime(true);
            
            // Check if bot is running
            $botRunning = file_exists('../bot_running.lock');
            
            // Perform detailed connection test
            $connectionTest = $this->performConnectionTest();
            
            $endTime = microtime(true);
            $responseTime = round(($endTime - $startTime) * 1000, 2);
            
            $connected = $botRunning && $connectionTest['success'];
            $status = $connected ? 'Connected' : 'Disconnected';
            $statusClass = $connected ? 'success' : 'danger';
            
            // Prepare detailed response
            $response = [
                'success' => true,
                'connected' => $connected,
                'status' => $status,
                'status_class' => $statusClass,
                'response_time' => $responseTime,
                'bot_running' => $botRunning,
                'timestamp' => date('Y-m-d H:i:s')
            ];
            
            // Add bot information if available
            if ($connectionTest['bot_info']) {
                $botInfo = $connectionTest['bot_info'];
                $response['bot_info'] = [
                    'id' => $botInfo['id'],
                    'username' => '@' . $botInfo['username'],
                    'name' => $botInfo['first_name'] . ($botInfo['last_name'] ? ' ' . $botInfo['last_name'] : ''),
                    'full_name' => $botInfo['first_name'] . ($botInfo['last_name'] ? ' ' . $botInfo['last_name'] : ''),
                    'can_join_groups' => $botInfo['can_join_groups'],
                    'can_read_all_group_messages' => $botInfo['can_read_all_group_messages'],
                    'supports_inline_queries' => $botInfo['supports_inline_queries'],
                    'is_bot' => $botInfo['is_bot']
                ];
            }
            
            // Add webhook information if available
            if ($connectionTest['webhook_info']) {
                $webhookInfo = $connectionTest['webhook_info'];
                $response['webhook_info'] = [
                    'url' => $webhookInfo['url'],
                    'has_custom_certificate' => $webhookInfo['has_custom_certificate'],
                    'pending_update_count' => $webhookInfo['pending_update_count'],
                    'max_connections' => $webhookInfo['max_connections'],
                    'last_error_date' => $webhookInfo['last_error_date'],
                    'last_error_message' => $webhookInfo['last_error_message'],
                    'allowed_updates' => $webhookInfo['allowed_updates']
                ];
            }
            
            // Add error information if any
            if ($connectionTest['error']) {
                $response['error'] = $connectionTest['error'];
            }
            
            $this->logger->log('INFO', "Bot connection check: $status (Response time: {$responseTime}ms)");
            
            return $response;
            
        } catch (Exception $e) {
            $this->logger->log('ERROR', "Failed to check bot connection: " . $e->getMessage());
            return [
                'success' => false,
                'connected' => false,
                'status' => 'Error',
                'status_class' => 'danger',
                'response_time' => 0,
                'error' => $e->getMessage(),
                'timestamp' => date('Y-m-d H:i:s')
            ];
        }
    }
    
    /**
     * Perform connection test
     */
    private function performConnectionTest(): array {
        try {
            $result = [
                'success' => false,
                'bot_info' => null,
                'webhook_info' => null,
                'error' => null
            ];
            
            // Method 1: Check if bot file exists and is accessible
            $botFile = '../telegram_bot.php';
            if (!file_exists($botFile)) {
                $result['error'] = 'Bot file not found';
                return $result;
            }
            
            // Method 2: Get bot configuration
            $botConfig = $this->getBotConfiguration();
            if (!$botConfig) {
                $result['error'] = 'Bot configuration not found';
                return $result;
            }
            
            // Method 3: Test Telegram API connection
            $botInfo = $this->testTelegramAPI($botConfig);
            if ($botInfo) {
                $result['success'] = true;
                $result['bot_info'] = $botInfo;
            } else {
                $result['error'] = 'Failed to connect to Telegram API';
            }
            
            // Method 4: Check webhook status
            $webhookInfo = $this->checkWebhookStatus($botConfig);
            $result['webhook_info'] = $webhookInfo;
            
            return $result;
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'bot_info' => null,
                'webhook_info' => null,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Get bot configuration
     */
    private function getBotConfiguration(): ?array {
        try {
            // Try to get bot token from config files
            $configFiles = [
                '../../config.php',
                '../../.env',
                '../../config/bot_config.php',
                '../../telegram_bot.php'
            ];
            
            foreach ($configFiles as $file) {
                if (file_exists($file)) {
                    $content = file_get_contents($file);
                    
                    // Look for bot token patterns with more flexible regex
                    $patterns = [
                        '/BOT_TOKEN["\']?\s*[=:]\s*["\']?([0-9]+:[A-Za-z0-9_-]+)["\']?/i',
                        '/TELEGRAM_BOT_TOKEN["\']?\s*[=:]\s*["\']?([0-9]+:[A-Za-z0-9_-]+)["\']?/i',
                        '/\$botToken\s*=\s*["\']([0-9]+:[A-Za-z0-9_-]+)["\']/i',
                        '/define\s*\(\s*["\']BOT_TOKEN["\']\s*,\s*["\']([0-9]+:[A-Za-z0-9_-]+)["\']/i',
                        '/["\']([0-9]+:[A-Za-z0-9_-]+)["\']/', // Generic pattern for bot token format
                    ];
                    
                    foreach ($patterns as $pattern) {
                        if (preg_match($pattern, $content, $matches)) {
                            $token = $matches[1];
                            // Validate token format (should be number:alphanumeric)
                            if (preg_match('/^[0-9]+:[A-Za-z0-9_-]+$/', $token)) {
                                error_log("Bot token found in $file: " . substr($token, 0, 10) . "...");
                                return ['token' => $token];
                            }
                        }
                    }
                }
            }
            
            // Try to get from database
            try {
                $stmt = $this->pdo->query("SELECT setting_value FROM bot_settings WHERE setting_name = 'bot_token'");
                $row = $stmt->fetch();
                if ($row && !empty($row['setting_value'])) {
                    $token = $row['setting_value'];
                    if (preg_match('/^[0-9]+:[A-Za-z0-9_-]+$/', $token)) {
                        error_log("Bot token found in database: " . substr($token, 0, 10) . "...");
                        return ['token' => $token];
                    }
                }
            } catch (Exception $e) {
                // Database might not have the table yet
            }
            
            error_log("No valid bot token found in any config files or database");
            return null;
            
        } catch (Exception $e) {
            error_log("Error getting bot configuration: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Test Telegram API connection
     */
    private function testTelegramAPI(array $config): ?array {
        try {
            if (empty($config['token'])) {
                error_log("No bot token provided for API test");
                return null;
            }
            
            $token = $config['token'];
            
            // Validate token format
            if (!preg_match('/^[0-9]+:[A-Za-z0-9_-]+$/', $token)) {
                error_log("Invalid bot token format: " . substr($token, 0, 10) . "...");
                return null;
            }
            
            $url = "https://api.telegram.org/bot{$token}/getMe";
            error_log("Testing Telegram API with URL: " . substr($url, 0, 50) . "...");
            
            $context = stream_context_create([
                'http' => [
                    'timeout' => 10,
                    'method' => 'GET',
                    'header' => 'User-Agent: AdminPanel/1.0',
                    'ignore_errors' => true
                ]
            ]);
            
            $response = file_get_contents($url, false, $context);
            if ($response === false) {
                error_log("Failed to get response from Telegram API");
                return null;
            }
            
            $data = json_decode($response, true);
            if (!$data) {
                error_log("Failed to decode JSON response from Telegram API");
                return null;
            }
            
            if (!$data['ok']) {
                error_log("Telegram API error: " . ($data['description'] ?? 'Unknown error'));
                return null;
            }
            
            error_log("Successfully connected to Telegram API for bot: " . $data['result']['username']);
            
            return [
                'id' => $data['result']['id'],
                'username' => $data['result']['username'],
                'first_name' => $data['result']['first_name'],
                'last_name' => $data['result']['last_name'] ?? '',
                'can_join_groups' => $data['result']['can_join_groups'],
                'can_read_all_group_messages' => $data['result']['can_read_all_group_messages'],
                'supports_inline_queries' => $data['result']['supports_inline_queries'],
                'is_bot' => $data['result']['is_bot']
            ];
            
        } catch (Exception $e) {
            error_log("Exception in testTelegramAPI: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Check webhook status
     */
    private function checkWebhookStatus(array $config): ?array {
        try {
            if (empty($config['token'])) {
                error_log("No bot token provided for webhook check");
                return null;
            }
            
            $token = $config['token'];
            
            // Validate token format
            if (!preg_match('/^[0-9]+:[A-Za-z0-9_-]+$/', $token)) {
                error_log("Invalid bot token format for webhook check: " . substr($token, 0, 10) . "...");
                return null;
            }
            
            $url = "https://api.telegram.org/bot{$token}/getWebhookInfo";
            error_log("Checking webhook status with URL: " . substr($url, 0, 50) . "...");
            
            $context = stream_context_create([
                'http' => [
                    'timeout' => 10,
                    'method' => 'GET',
                    'header' => 'User-Agent: AdminPanel/1.0',
                    'ignore_errors' => true
                ]
            ]);
            
            $response = file_get_contents($url, false, $context);
            if ($response === false) {
                error_log("Failed to get webhook info from Telegram API");
                return null;
            }
            
            $data = json_decode($response, true);
            if (!$data) {
                error_log("Failed to decode JSON response from webhook API");
                return null;
            }
            
            if (!$data['ok']) {
                error_log("Telegram webhook API error: " . ($data['description'] ?? 'Unknown error'));
                return null;
            }
            
            error_log("Successfully retrieved webhook info");
            
            return [
                'url' => $data['result']['url'],
                'has_custom_certificate' => $data['result']['has_custom_certificate'],
                'pending_update_count' => $data['result']['pending_update_count'],
                'last_error_date' => $data['result']['last_error_date'],
                'last_error_message' => $data['result']['last_error_message'],
                'max_connections' => $data['result']['max_connections'],
                'allowed_updates' => $data['result']['allowed_updates']
            ];
            
        } catch (Exception $e) {
            error_log("Exception in checkWebhookStatus: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Get log settings
     */
    public function getLogSettings(): array {
        try {
            $stmt = $this->pdo->query("SELECT * FROM log_settings");
            $settings = [];
            while ($row = $stmt->fetch()) {
                $settings[$row['setting_name']] = $row['setting_value'];
            }
            return $settings;
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Failed to get log settings: ' . $e->getMessage());
            return [];
        }
    }
}