<?php
/**
 * Authentication Manager
 * Handles admin authentication and session management
 */

class AuthManager {
    private $pdo;
    
    public function __construct(PDO $pdo) {
        $this->pdo = $pdo;
    }
    
    /**
     * Check if user is authenticated
     */
    public function isAuthenticated(): bool {
        return isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in'] === true;
    }
    
    /**
     * Login user
     */
    public function login(string $username, string $password): bool {
        try {
            $stmt = $this->pdo->prepare("
                SELECT id, username, password, role, is_active, failed_attempts, last_failed_attempt 
                FROM admin_users 
                WHERE username = ? AND is_active = 1
            ");
            $stmt->execute([$username]);
            $user = $stmt->fetch();
            
            if (!$user) {
                return false;
            }
            
            // Check if account is locked
            if ($user['failed_attempts'] >= ADMIN_MAX_LOGIN_ATTEMPTS) {
                $lockoutTime = strtotime($user['last_failed_attempt']) + ADMIN_LOCKOUT_TIME;
                if (time() < $lockoutTime) {
                    return false;
                }
            }
            
            // Verify password
            if (password_verify($password, $user['password'])) {
                // Reset failed attempts
                $this->resetFailedAttempts($user['id']);
                
                // Update last login
                $this->updateLastLogin($user['id']);
                
                // Set session
                $_SESSION['admin_logged_in'] = true;
                $_SESSION['admin_user_id'] = $user['id'];
                $_SESSION['admin_username'] = $user['username'];
                $_SESSION['admin_role'] = $user['role'];
                
                return true;
            } else {
                // Increment failed attempts
                $this->incrementFailedAttempts($user['id']);
                return false;
            }
        } catch (Exception $e) {
            error_log("Login error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Logout user
     */
    public function logout(): void {
        session_destroy();
    }
    
    /**
     * Get current admin username
     */
    public function getAdminUsername(): ?string {
        return $_SESSION['admin_username'] ?? null;
    }
    
    /**
     * Get current admin role
     */
    public function getAdminRole(): ?string {
        return $_SESSION['admin_role'] ?? null;
    }
    
    /**
     * Get current admin ID
     */
    public function getAdminId(): ?int {
        return $_SESSION['admin_user_id'] ?? null;
    }
    
    /**
     * Check if user has permission
     */
    public function hasPermission(string $permission): bool {
        $role = $this->getAdminRole();
        
        switch ($role) {
            case 'super_admin':
                return true;
            case 'admin':
                return in_array($permission, ['users', 'logs', 'settings', 'broadcast']);
            case 'moderator':
                return in_array($permission, ['users', 'logs']);
            default:
                return false;
        }
    }
    
    /**
     * Reset failed login attempts
     */
    private function resetFailedAttempts(int $userId): void {
        try {
            $stmt = $this->pdo->prepare("
                UPDATE admin_users 
                SET failed_attempts = 0, last_failed_attempt = NULL 
                WHERE id = ?
            ");
            $stmt->execute([$userId]);
        } catch (Exception $e) {
            error_log("Reset failed attempts error: " . $e->getMessage());
        }
    }
    
    /**
     * Increment failed login attempts
     */
    private function incrementFailedAttempts(int $userId): void {
        try {
            $stmt = $this->pdo->prepare("
                UPDATE admin_users 
                SET failed_attempts = failed_attempts + 1, last_failed_attempt = NOW() 
                WHERE id = ?
            ");
            $stmt->execute([$userId]);
        } catch (Exception $e) {
            error_log("Increment failed attempts error: " . $e->getMessage());
        }
    }
    
    /**
     * Update last login time
     */
    private function updateLastLogin(int $userId): void {
        try {
            $stmt = $this->pdo->prepare("
                UPDATE admin_users 
                SET last_login = NOW() 
                WHERE id = ?
            ");
            $stmt->execute([$userId]);
        } catch (Exception $e) {
            error_log("Update last login error: " . $e->getMessage());
        }
    }
    
    /**
     * Change password
     */
    public function changePassword(string $currentPassword, string $newPassword): bool {
        if (!$this->isAuthenticated()) {
            return false;
        }
        
        try {
            $userId = $this->getAdminId();
            $stmt = $this->pdo->prepare("SELECT password FROM admin_users WHERE id = ?");
            $stmt->execute([$userId]);
            $user = $stmt->fetch();
            
            if (!$user || !password_verify($currentPassword, $user['password'])) {
                return false;
            }
            
            $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
            $stmt = $this->pdo->prepare("UPDATE admin_users SET password = ? WHERE id = ?");
            $stmt->execute([$hashedPassword, $userId]);
            
            return true;
        } catch (Exception $e) {
            error_log("Change password error: " . $e->getMessage());
            return false;
        }
    }
}