<?php
/**
 * Bot Manager
 * Handles bot configuration and control
 */

class BotManager {
    private $pdo;
    private $logger;
    
    public function __construct(PDO $pdo, DatabaseLogger $logger) {
        $this->pdo = $pdo;
        $this->logger = $logger;
    }
    
    /**
     * Get bot status
     */
    public function getBotStatus(): array {
        try {
            // Check if bot is running (simple file-based check)
            $botRunning = file_exists('../bot_running.lock');
            
            // Get bot settings
            $settings = $this->getBotSettings();
            
            // Get bot commands
            $commands = $this->getBotCommands();
            
            // Get recent bot logs
            $recentLogs = [];
            try {
                $stmt = $this->pdo->prepare("
                    SELECT * FROM bot_logs 
                    WHERE log_type = 'bot_status' 
                    ORDER BY created_at DESC 
                    LIMIT 10
                ");
                $stmt->execute();
                $recentLogs = $stmt->fetchAll();
            } catch (Exception $e) {
                // If bot_logs table doesn't exist, just return empty array
                $recentLogs = [];
            }
            
            return [
                'success' => true,
                'running' => $botRunning,
                'settings' => $settings,
                'commands' => $commands,
                'recent_logs' => $recentLogs,
                'last_check' => date('Y-m-d H:i:s'),
                'stats' => [
                    'total_commands' => count($commands),
                    'avg_response_time' => 150 // Mock data
                ]
            ];
        } catch (Exception $e) {
            if ($this->logger) {
                $this->logger->log('ERROR', 'Failed to get bot status: ' . $e->getMessage());
            }
            return [
                'success' => false,
                'running' => false,
                'settings' => [],
                'commands' => [],
                'recent_logs' => [],
                'error' => $e->getMessage(),
                'message' => 'Bot manager not available: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Get bot settings
     */
    public function getBotSettings(): array {
        try {
            $stmt = $this->pdo->query("SELECT * FROM bot_settings");
            $settings = [];
            while ($row = $stmt->fetch()) {
                $settings[$row['setting_name']] = [
                    'value' => $row['setting_value'],
                    'description' => $row['description']
                ];
            }
            return $settings;
        } catch (Exception $e) {
            if ($this->logger) {
                $this->logger->log('ERROR', 'Failed to get bot settings: ' . $e->getMessage());
            }
            // Return default settings if table doesn't exist
            return [
                'max_connections' => ['value' => '100', 'description' => 'Maximum webhook connections'],
                'rate_limit' => ['value' => '30', 'description' => 'Rate limit per minute per user'],
                'response_delay' => ['value' => '1', 'description' => 'Auto response delay in seconds'],
                'log_level' => ['value' => 'INFO', 'description' => 'Bot logging level'],
                'auto_restart' => ['value' => '1', 'description' => 'Auto restart on failure'],
                'maintenance_mode' => ['value' => '0', 'description' => 'Maintenance mode flag']
            ];
        }
    }
    
    /**
     * Update bot configuration
     */
    public function updateBotConfig(array $config): bool {
        try {
            $this->pdo->beginTransaction();
            
            foreach ($config as $key => $value) {
                $stmt = $this->pdo->prepare("
                    INSERT INTO bot_settings (setting_name, setting_value) 
                    VALUES (?, ?) 
                    ON DUPLICATE KEY UPDATE setting_value = ?
                ");
                $stmt->execute([$key, $value, $value]);
            }
            
            $this->pdo->commit();
            return true;
        } catch (Exception $e) {
            $this->pdo->rollBack();
            $this->logger->log('ERROR', 'Failed to update bot config: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get bot commands
     */
    public function getBotCommands(): array {
        try {
            $stmt = $this->pdo->query("SELECT * FROM bot_commands ORDER BY command_name");
            return $stmt->fetchAll();
        } catch (Exception $e) {
            if ($this->logger) {
                $this->logger->log('ERROR', 'Failed to get bot commands: ' . $e->getMessage());
            }
            // Return default commands if table doesn't exist
            return [
                [
                    'command_name' => 'start',
                    'is_enabled' => 1,
                    'rate_limit' => 30,
                    'description' => 'Start the bot'
                ],
                [
                    'command_name' => 'help',
                    'is_enabled' => 1,
                    'rate_limit' => 30,
                    'description' => 'Show help information'
                ],
                [
                    'command_name' => 'status',
                    'is_enabled' => 1,
                    'rate_limit' => 30,
                    'description' => 'Check bot status'
                ]
            ];
        }
    }
    
    /**
     * Toggle bot command
     */
    public function toggleBotCommand(string $command, bool $enabled): bool {
        try {
            $stmt = $this->pdo->prepare("
                UPDATE bot_commands 
                SET is_enabled = ? 
                WHERE command_name = ?
            ");
            $stmt->execute([$enabled ? 1 : 0, $command]);
            
            return $stmt->rowCount() > 0;
        } catch (Exception $e) {
            if ($this->logger) {
                $this->logger->log('ERROR', 'Failed to toggle bot command: ' . $e->getMessage());
            }
            return false;
        }
    }
    
    /**
     * Update command rate limit
     */
    public function updateCommandRateLimit(string $command, int $rateLimit): bool {
        try {
            $stmt = $this->pdo->prepare("
                UPDATE bot_commands 
                SET rate_limit = ? 
                WHERE command_name = ?
            ");
            $stmt->execute([$rateLimit, $command]);
            
            return $stmt->rowCount() > 0;
        } catch (Exception $e) {
            if ($this->logger) {
                $this->logger->log('ERROR', 'Failed to update command rate limit: ' . $e->getMessage());
            }
            return false;
        }
    }
    
    /**
     * Start bot
     */
    public function startBot(): bool {
        try {
            // Create lock file to indicate bot is running
            file_put_contents('../bot_running.lock', date('Y-m-d H:i:s'));
            
            // Log the action
            $this->logger->log('INFO', 'Bot started by admin');
            
            return true;
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Failed to start bot: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Stop bot
     */
    public function stopBot(): bool {
        try {
            // Remove lock file to indicate bot is stopped
            if (file_exists('../bot_running.lock')) {
                unlink('../bot_running.lock');
            }
            
            // Log the action
            $this->logger->log('INFO', 'Bot stopped by admin');
            
            return true;
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Failed to stop bot: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Restart bot
     */
    public function restartBot(): bool {
        try {
            // Stop first
            $this->stopBot();
            
            // Wait a moment
            sleep(2);
            
            // Start again
            $this->startBot();
            
            // Log the action
            $this->logger->log('INFO', 'Bot restarted by admin');
            
            return true;
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Failed to restart bot: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get bot statistics
     */
    public function getBotStats(): array {
        try {
            // Get total commands executed today
            $stmt = $this->pdo->prepare("
                SELECT COUNT(*) as total_commands 
                FROM bot_logs 
                WHERE log_type = 'command' 
                AND DATE(created_at) = CURDATE()
            ");
            $stmt->execute();
            $totalCommands = $stmt->fetchColumn();
            
            // Get most used commands
            $stmt = $this->pdo->prepare("
                SELECT action, COUNT(*) as count 
                FROM bot_logs 
                WHERE log_type = 'command' 
                AND DATE(created_at) = CURDATE()
                GROUP BY action 
                ORDER BY count DESC 
                LIMIT 5
            ");
            $stmt->execute();
            $topCommands = $stmt->fetchAll();
            
            // Get response time average
            $stmt = $this->pdo->prepare("
                SELECT AVG(response_time) as avg_response_time 
                FROM bot_logs 
                WHERE response_time IS NOT NULL 
                AND DATE(created_at) = CURDATE()
            ");
            $stmt->execute();
            $avgResponseTime = $stmt->fetchColumn();
            
            return [
                'total_commands' => $totalCommands,
                'top_commands' => $topCommands,
                'avg_response_time' => round($avgResponseTime, 3)
            ];
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Failed to get bot stats: ' . $e->getMessage());
            return [
                'total_commands' => 0,
                'top_commands' => [],
                'avg_response_time' => 0
            ];
        }
    }
    
    /**
     * Add new command
     */
    public function addCommand(string $command, string $description, int $rateLimit = 30): bool {
        try {
            $stmt = $this->pdo->prepare("
                INSERT INTO bot_commands (command_name, description, rate_limit, is_enabled) 
                VALUES (?, ?, ?, 1)
            ");
            $stmt->execute([$command, $description, $rateLimit]);
            
            return true;
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Failed to add command: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Remove command
     */
    public function removeCommand(string $command): bool {
        try {
            $stmt = $this->pdo->prepare("DELETE FROM bot_commands WHERE command_name = ?");
            $stmt->execute([$command]);
            
            return $stmt->rowCount() > 0;
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Failed to remove command: ' . $e->getMessage());
            return false;
        }
    }
}