<?php
/**
 * Broadcast Manager
 * Handles broadcast messages and notifications
 */

class BroadcastManager {
    private $pdo;
    private $logger;
    
    public function __construct(PDO $pdo, DatabaseLogger $logger) {
        $this->pdo = $pdo;
        $this->logger = $logger;
    }
    
    /**
     * Create new broadcast
     */
    public function createBroadcast(string $message, ?string $scheduleTime = null, ?array $targetUsers = null): ?int {
        try {
            $this->pdo->beginTransaction();
            
            // Get admin ID
            $adminId = $_SESSION['admin_user_id'] ?? 1;
            
            // Determine target type
            $targetType = 'all';
            if ($targetUsers && count($targetUsers) > 0) {
                $targetType = 'specific';
            }
            
            // Insert broadcast
            $stmt = $this->pdo->prepare("
                INSERT INTO broadcast_logs (
                    admin_id, message, status, target_type, target_users, 
                    scheduled_at, priority, total_users
                ) VALUES (?, ?, 'pending', ?, ?, ?, 'normal', ?)
            ");
            
            $targetUsersJson = $targetUsers ? json_encode($targetUsers) : null;
            $totalUsers = $this->getTotalUsers($targetType, $targetUsers);
            
            $stmt->execute([
                $adminId,
                $message,
                $targetType,
                $targetUsersJson,
                $scheduleTime,
                $totalUsers
            ]);
            
            $broadcastId = $this->pdo->lastInsertId();
            
            // If no schedule time, start immediately
            if (!$scheduleTime) {
                $this->startBroadcast($broadcastId);
            }
            
            $this->pdo->commit();
            
            $this->logger->log('INFO', "Broadcast created: $broadcastId");
            
            return $broadcastId;
        } catch (Exception $e) {
            $this->pdo->rollBack();
            $this->logger->log('ERROR', 'Failed to create broadcast: ' . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Get broadcasts
     */
    public function getBroadcasts(int $limit = 50, int $offset = 0): array {
        try {
            $stmt = $this->pdo->prepare("
                SELECT bl.*, au.username as admin_username
                FROM broadcast_logs bl
                LEFT JOIN admin_users au ON bl.admin_id = au.id
                ORDER BY bl.created_at DESC 
                LIMIT ? OFFSET ?
            ");
            $stmt->execute([$limit, $offset]);
            return $stmt->fetchAll();
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Failed to get broadcasts: ' . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get broadcast details
     */
    public function getBroadcastDetails(int $broadcastId): ?array {
        try {
            $stmt = $this->pdo->prepare("
                SELECT bl.*, au.username as admin_username
                FROM broadcast_logs bl
                LEFT JOIN admin_users au ON bl.admin_id = au.id
                WHERE bl.id = ?
            ");
            $stmt->execute([$broadcastId]);
            return $stmt->fetch();
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Failed to get broadcast details: ' . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Update broadcast status
     */
    public function updateBroadcastStatus(int $broadcastId, string $status, int $successCount = 0, int $failedCount = 0): bool {
        try {
            $stmt = $this->pdo->prepare("
                UPDATE broadcast_logs 
                SET status = ?, success_count = ?, failed_count = ?, 
                    started_at = CASE WHEN ? = 'running' AND started_at IS NULL THEN NOW() ELSE started_at END,
                    completed_at = CASE WHEN ? IN ('completed', 'failed', 'cancelled') THEN NOW() ELSE completed_at END
                WHERE id = ?
            ");
            $stmt->execute([$status, $successCount, $failedCount, $status, $status, $broadcastId]);
            
            return $stmt->rowCount() > 0;
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Failed to update broadcast status: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Start broadcast
     */
    public function startBroadcast(int $broadcastId): bool {
        try {
            $broadcast = $this->getBroadcastDetails($broadcastId);
            if (!$broadcast) {
                return false;
            }
            
            // Update status to running
            $this->updateBroadcastStatus($broadcastId, 'running');
            
            // Get target users
            $users = $this->getTargetUsers($broadcast['target_type'], $broadcast['target_users']);
            
            // Process broadcast in background
            $this->processBroadcast($broadcastId, $broadcast['message'], $users);
            
            return true;
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Failed to start broadcast: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Process broadcast (simplified version)
     */
    private function processBroadcast(int $broadcastId, string $message, array $users): void {
        try {
            $successCount = 0;
            $failedCount = 0;
            
            foreach ($users as $user) {
                // Simulate sending message (replace with actual Telegram API call)
                $success = $this->sendMessageToUser($user['id'], $message);
                
                if ($success) {
                    $successCount++;
                } else {
                    $failedCount++;
                }
                
                // Add small delay to avoid rate limiting
                usleep(100000); // 0.1 second
            }
            
            // Update final status
            $this->updateBroadcastStatus($broadcastId, 'completed', $successCount, $failedCount);
            
            $this->logger->log('INFO', "Broadcast $broadcastId completed: $successCount success, $failedCount failed");
            
        } catch (Exception $e) {
            $this->updateBroadcastStatus($broadcastId, 'failed', 0, count($users));
            $this->logger->log('ERROR', 'Failed to process broadcast: ' . $e->getMessage());
        }
    }
    
    /**
     * Send message to user (placeholder)
     */
    private function sendMessageToUser(int $userId, string $message): bool {
        // This is a placeholder - implement actual Telegram API call here
        // For now, just simulate success/failure
        return rand(1, 10) > 2; // 80% success rate
    }
    
    /**
     * Get target users
     */
    private function getTargetUsers(string $targetType, ?string $targetUsersJson): array {
        try {
            if ($targetType === 'specific' && $targetUsersJson) {
                $targetUserIds = json_decode($targetUsersJson, true);
                if (is_array($targetUserIds)) {
                    $placeholders = str_repeat('?,', count($targetUserIds) - 1) . '?';
                    $stmt = $this->pdo->prepare("
                        SELECT id, username, first_name, last_name 
                        FROM users 
                        WHERE id IN ($placeholders)
                    ");
                    $stmt->execute($targetUserIds);
                    return $stmt->fetchAll();
                }
            }
            
            // Get all users or active users
            if ($targetType === 'active') {
                $stmt = $this->pdo->prepare("
                    SELECT id, username, first_name, last_name 
                    FROM users 
                    WHERE last_seen >= DATE_SUB(NOW(), INTERVAL 7 DAY)
                    ORDER BY last_seen DESC
                ");
            } else {
                $stmt = $this->pdo->prepare("
                    SELECT id, username, first_name, last_name 
                    FROM users 
                    ORDER BY created_at DESC
                ");
            }
            
            $stmt->execute();
            return $stmt->fetchAll();
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Failed to get target users: ' . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get total users count
     */
    private function getTotalUsers(string $targetType, ?array $targetUsers): int {
        try {
            if ($targetType === 'specific' && $targetUsers) {
                return count($targetUsers);
            }
            
            if ($targetType === 'active') {
                $stmt = $this->pdo->prepare("
                    SELECT COUNT(*) 
                    FROM users 
                    WHERE last_seen >= DATE_SUB(NOW(), INTERVAL 7 DAY)
                ");
            } else {
                $stmt = $this->pdo->prepare("SELECT COUNT(*) FROM users");
            }
            
            $stmt->execute();
            return $stmt->fetchColumn();
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Failed to get total users: ' . $e->getMessage());
            return 0;
        }
    }
    
    /**
     * Cancel broadcast
     */
    public function cancelBroadcast(int $broadcastId): bool {
        try {
            $stmt = $this->pdo->prepare("
                UPDATE broadcast_logs 
                SET status = 'cancelled', cancelled_at = NOW(), cancelled_by = ?
                WHERE id = ? AND status IN ('pending', 'running')
            ");
            $stmt->execute([$_SESSION['admin_user_id'] ?? 1, $broadcastId]);
            
            return $stmt->rowCount() > 0;
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Failed to cancel broadcast: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get broadcast statistics
     */
    public function getBroadcastStats(): array {
        try {
            // Total broadcasts
            $stmt = $this->pdo->query("SELECT COUNT(*) FROM broadcast_logs");
            $totalBroadcasts = $stmt->fetchColumn();
            
            // Successful broadcasts
            $stmt = $this->pdo->query("SELECT COUNT(*) FROM broadcast_logs WHERE status = 'completed'");
            $successfulBroadcasts = $stmt->fetchColumn();
            
            // Total messages sent
            $stmt = $this->pdo->query("SELECT SUM(success_count) FROM broadcast_logs WHERE status = 'completed'");
            $totalMessagesSent = $stmt->fetchColumn() ?: 0;
            
            // Recent broadcasts (last 7 days)
            $stmt = $this->pdo->prepare("
                SELECT COUNT(*) 
                FROM broadcast_logs 
                WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)
            ");
            $stmt->execute();
            $recentBroadcasts = $stmt->fetchColumn();
            
            return [
                'total_broadcasts' => $totalBroadcasts,
                'successful_broadcasts' => $successfulBroadcasts,
                'total_messages_sent' => $totalMessagesSent,
                'recent_broadcasts' => $recentBroadcasts
            ];
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Failed to get broadcast stats: ' . $e->getMessage());
            return [
                'total_broadcasts' => 0,
                'successful_broadcasts' => 0,
                'total_messages_sent' => 0,
                'recent_broadcasts' => 0
            ];
        }
    }
}