<?php
/**
 * Telegram Integration for Admin Panel
 * Handles communication between admin panel and Telegram bot
 */

class TelegramIntegration {
    private $botToken;
    private $logger;
    private $config;
    
    public function __construct($botToken, $logger, $config = []) {
        $this->botToken = $botToken;
        $this->logger = $logger;
        $this->config = $config;
    }
    
    /**
     * Send notification to admin
     */
    public function sendAdminNotification($message, $type = 'info') {
        try {
            $adminId = $this->config['admin_telegram_id'] ?? null;
            if (!$adminId) {
                return false;
            }
            
            $icon = $this->getNotificationIcon($type);
            $formattedMessage = "{$icon} <b>Admin Notification</b>\n\n{$message}";
            
            return $this->sendMessage($adminId, $formattedMessage);
            
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Failed to send admin notification', [
                'error' => $e->getMessage(),
                'type' => $type
            ]);
            return false;
        }
    }
    
    /**
     * Send broadcast status update
     */
    public function sendBroadcastStatus($adminId, $broadcastId, $status, $details = []) {
        try {
            $message = $this->formatBroadcastStatusMessage($broadcastId, $status, $details);
            return $this->sendMessage($adminId, $message);
            
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Failed to send broadcast status', [
                'error' => $e->getMessage(),
                'broadcast_id' => $broadcastId
            ]);
            return false;
        }
    }
    
    /**
     * Send bot status update
     */
    public function sendBotStatusUpdate($adminId, $status, $details = []) {
        try {
            $message = $this->formatBotStatusMessage($status, $details);
            return $this->sendMessage($adminId, $message);
            
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Failed to send bot status update', [
                'error' => $e->getMessage(),
                'status' => $status
            ]);
            return false;
        }
    }
    
    /**
     * Send system alert
     */
    public function sendSystemAlert($message, $level = 'warning') {
        try {
            $adminId = $this->config['admin_telegram_id'] ?? null;
            if (!$adminId) {
                return false;
            }
            
            $icon = $this->getAlertIcon($level);
            $formattedMessage = "{$icon} <b>System Alert</b>\n\n{$message}";
            
            return $this->sendMessage($adminId, $formattedMessage);
            
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Failed to send system alert', [
                'error' => $e->getMessage(),
                'level' => $level
            ]);
            return false;
        }
    }
    
    /**
     * Get bot information
     */
    public function getBotInfo() {
        try {
            $url = "https://api.telegram.org/bot{$this->botToken}/getMe";
            $response = file_get_contents($url);
            $data = json_decode($response, true);
            
            if ($data['ok']) {
                return $data['result'];
            } else {
                throw new Exception($data['description'] ?? 'Unknown error');
            }
            
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Failed to get bot info', [
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }
    
    /**
     * Get webhook info
     */
    public function getWebhookInfo() {
        try {
            $url = "https://api.telegram.org/bot{$this->botToken}/getWebhookInfo";
            $response = file_get_contents($url);
            $data = json_decode($response, true);
            
            if ($data['ok']) {
                return $data['result'];
            } else {
                throw new Exception($data['description'] ?? 'Unknown error');
            }
            
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Failed to get webhook info', [
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }
    
    /**
     * Set webhook
     */
    public function setWebhook($url, $maxConnections = 100) {
        try {
            $apiUrl = "https://api.telegram.org/bot{$this->botToken}/setWebhook";
            $data = [
                'url' => $url,
                'max_connections' => $maxConnections,
                'allowed_updates' => ['message', 'callback_query']
            ];
            
            $response = $this->makeRequest($apiUrl, $data);
            
            if ($response['ok']) {
                $this->logger->log('INFO', 'Webhook set successfully', [
                    'url' => $url,
                    'max_connections' => $maxConnections
                ]);
                return true;
            } else {
                throw new Exception($response['description'] ?? 'Unknown error');
            }
            
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Failed to set webhook', [
                'error' => $e->getMessage(),
                'url' => $url
            ]);
            return false;
        }
    }
    
    /**
     * Delete webhook
     */
    public function deleteWebhook() {
        try {
            $apiUrl = "https://api.telegram.org/bot{$this->botToken}/deleteWebhook";
            $response = $this->makeRequest($apiUrl);
            
            if ($response['ok']) {
                $this->logger->log('INFO', 'Webhook deleted successfully');
                return true;
            } else {
                throw new Exception($response['description'] ?? 'Unknown error');
            }
            
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Failed to delete webhook', [
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }
    
    /**
     * Send message to user
     */
    private function sendMessage($chatId, $message, $parseMode = 'HTML') {
        try {
            $url = "https://api.telegram.org/bot{$this->botToken}/sendMessage";
            $data = [
                'chat_id' => $chatId,
                'text' => $message,
                'parse_mode' => $parseMode
            ];
            
            $response = $this->makeRequest($url, $data);
            
            if ($response['ok']) {
                return $response['result'];
            } else {
                throw new Exception($response['description'] ?? 'Unknown error');
            }
            
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Failed to send message', [
                'error' => $e->getMessage(),
                'chat_id' => $chatId
            ]);
            return false;
        }
    }
    
    /**
     * Make HTTP request
     */
    private function makeRequest($url, $data = []) {
        $options = [
            'http' => [
                'header' => "Content-type: application/x-www-form-urlencoded\r\n",
                'method' => 'POST',
                'content' => http_build_query($data)
            ]
        ];
        
        $context = stream_context_create($options);
        $response = file_get_contents($url, false, $context);
        
        return json_decode($response, true);
    }
    
    /**
     * Format broadcast status message
     */
    private function formatBroadcastStatusMessage($broadcastId, $status, $details) {
        $icon = $this->getBroadcastIcon($status);
        $message = "{$icon} <b>Broadcast Status Update</b>\n\n";
        $message .= "🆔 <b>ID:</b> {$broadcastId}\n";
        $message .= "📊 <b>Status:</b> " . ucfirst($status) . "\n";
        
        if (isset($details['total_users'])) {
            $message .= "👥 <b>Target Users:</b> {$details['total_users']}\n";
        }
        
        if (isset($details['sent'])) {
            $message .= "✅ <b>Sent:</b> {$details['sent']}\n";
        }
        
        if (isset($details['failed'])) {
            $message .= "❌ <b>Failed:</b> {$details['failed']}\n";
        }
        
        if (isset($details['progress'])) {
            $message .= "📈 <b>Progress:</b> {$details['progress']}%\n";
        }
        
        $message .= "⏰ <b>Time:</b> " . date('H:i:s') . " WIB";
        
        return $message;
    }
    
    /**
     * Format bot status message
     */
    private function formatBotStatusMessage($status, $details) {
        $icon = $this->getBotIcon($status);
        $message = "{$icon} <b>Bot Status Update</b>\n\n";
        $message .= "🤖 <b>Status:</b> " . ucfirst($status) . "\n";
        
        if (isset($details['uptime'])) {
            $message .= "⏱️ <b>Uptime:</b> {$details['uptime']}\n";
        }
        
        if (isset($details['requests'])) {
            $message .= "📊 <b>Today's Requests:</b> {$details['requests']}\n";
        }
        
        if (isset($details['error_rate'])) {
            $message .= "⚠️ <b>Error Rate:</b> {$details['error_rate']}%\n";
        }
        
        $message .= "⏰ <b>Time:</b> " . date('H:i:s') . " WIB";
        
        return $message;
    }
    
    /**
     * Get notification icon
     */
    private function getNotificationIcon($type) {
        switch ($type) {
            case 'success': return '✅';
            case 'warning': return '⚠️';
            case 'error': return '❌';
            case 'info': return 'ℹ️';
            default: return '📢';
        }
    }
    
    /**
     * Get alert icon
     */
    private function getAlertIcon($level) {
        switch ($level) {
            case 'critical': return '🚨';
            case 'error': return '❌';
            case 'warning': return '⚠️';
            case 'info': return 'ℹ️';
            default: return '📢';
        }
    }
    
    /**
     * Get broadcast icon
     */
    private function getBroadcastIcon($status) {
        switch ($status) {
            case 'completed': return '✅';
            case 'running': return '🔄';
            case 'pending': return '⏳';
            case 'cancelled': return '❌';
            case 'failed': return '💥';
            default: return '📢';
        }
    }
    
    /**
     * Get bot icon
     */
    private function getBotIcon($status) {
        switch ($status) {
            case 'online': return '🟢';
            case 'offline': return '🔴';
            case 'error': return '⚠️';
            case 'maintenance': return '🔧';
            default: return '🤖';
        }
    }
}
