<?php
/**
 * Production Setup Script
 * Run this to prepare the system for production
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

echo "<h1>🚀 Production Setup</h1>\n";
echo "<p>Setting up the system for production environment...</p>\n";

$steps = [];
$errors = [];

// Step 1: Check database connection
echo "<h2>Step 1: Checking Database Connection</h2>\n";
try {
    require_once '../config/database.php';
    $database = new Database();
    $pdo = $database->getConnection();
    $steps[] = ['status' => 'success', 'message' => 'Database connection successful'];
    echo "✅ Database connection successful<br>\n";
} catch (Exception $e) {
    $steps[] = ['status' => 'error', 'message' => 'Database connection failed: ' . $e->getMessage()];
    $errors[] = $e->getMessage();
    echo "❌ Database connection failed: " . $e->getMessage() . "<br>\n";
}

// Step 2: Check required tables
echo "<br><h2>Step 2: Checking Required Tables</h2>\n";
$requiredTables = [
    'bot_settings',
    'bot_commands',
    'bot_logs',
    'admin_users',
    'broadcasts',
    'broadcast_logs'
];

foreach ($requiredTables as $table) {
    try {
        $stmt = $pdo->query("SHOW TABLES LIKE '$table'");
        $exists = $stmt->rowCount() > 0;
        
        if ($exists) {
            $steps[] = ['status' => 'success', 'message' => "Table '$table' exists"];
            echo "✅ Table '$table' exists<br>\n";
        } else {
            $steps[] = ['status' => 'warning', 'message' => "Table '$table' does not exist"];
            echo "⚠️ Table '$table' does not exist<br>\n";
        }
    } catch (Exception $e) {
        $steps[] = ['status' => 'error', 'message' => "Error checking table '$table': " . $e->getMessage()];
        $errors[] = $e->getMessage();
        echo "❌ Error checking table '$table': " . $e->getMessage() . "<br>\n";
    }
}

// Step 3: Verify bot token
echo "<br><h2>Step 3: Verifying Bot Token</h2>\n";
try {
    $stmt = $pdo->query("SELECT setting_value FROM bot_settings WHERE setting_name = 'bot_token'");
    $botToken = $stmt->fetchColumn();
    
    if ($botToken) {
        echo "✅ Bot token found in database: " . substr($botToken, 0, 10) . "...<br>\n";
        
        // Test the token
        $url = "https://api.telegram.org/bot{$botToken}/getMe";
        $context = stream_context_create([
            'http' => [
                'timeout' => 10,
                'method' => 'GET',
                'header' => 'User-Agent: AdminPanel/1.0',
                'ignore_errors' => true
            ]
        ]);
        
        $response = file_get_contents($url, false, $context);
        if ($response !== false) {
            $data = json_decode($response, true);
            if ($data && $data['ok']) {
                $steps[] = ['status' => 'success', 'message' => 'Bot token is valid'];
                echo "✅ Bot token is valid! Bot: @" . $data['result']['username'] . " (" . $data['result']['first_name'] . ")<br>\n";
            } else {
                $steps[] = ['status' => 'error', 'message' => 'Bot token is invalid'];
                $errors[] = 'Bot token is invalid';
                echo "❌ Bot token is invalid: " . ($data['description'] ?? 'Unknown error') . "<br>\n";
            }
        } else {
            $steps[] = ['status' => 'warning', 'message' => 'Could not test bot token (network error)'];
            echo "⚠️ Could not test bot token (network error)<br>\n";
        }
    } else {
        $steps[] = ['status' => 'error', 'message' => 'Bot token not found in database'];
        $errors[] = 'Bot token not found';
        echo "❌ Bot token not found in database<br>\n";
    }
} catch (Exception $e) {
    $steps[] = ['status' => 'error', 'message' => 'Error verifying bot token: ' . $e->getMessage()];
    $errors[] = $e->getMessage();
    echo "❌ Error verifying bot token: " . $e->getMessage() . "<br>\n";
}

// Step 4: Check file permissions
echo "<br><h2>Step 4: Checking File Permissions</h2>\n";
$criticalFiles = [
    '../telegram_bot.php',
    '../config/database.php',
    '../classes',
    '../logs'
];

foreach ($criticalFiles as $file) {
    if (file_exists($file)) {
        $permissions = substr(sprintf('%o', fileperms($file)), -4);
        echo "✅ $file: permissions $permissions<br>\n";
        $steps[] = ['status' => 'success', 'message' => "File '$file' exists with permissions $permissions"];
    } else {
        echo "❌ $file: does not exist<br>\n";
        $steps[] = ['status' => 'error', 'message' => "File '$file' does not exist"];
        $errors[] = "File '$file' does not exist";
    }
}

// Step 5: Summary
echo "<br><h2>Step 5: Summary</h2>\n";
$successCount = count(array_filter($steps, function($step) { return $step['status'] === 'success'; }));
$warningCount = count(array_filter($steps, function($step) { return $step['status'] === 'warning'; }));
$errorCount = count(array_filter($steps, function($step) { return $step['status'] === 'error'; }));

echo "<p>Total checks: " . count($steps) . "</p>\n";
echo "<p>✅ Successful: $successCount</p>\n";
echo "<p>⚠️ Warnings: $warningCount</p>\n";
echo "<p>❌ Errors: $errorCount</p>\n";

if (empty($errors)) {
    echo "<br><h2 style='color: green;'>✅ System is ready for production!</h2>\n";
    echo "<p>All critical checks passed. The system is ready to use.</p>\n";
} else {
    echo "<br><h2 style='color: red;'>❌ System is NOT ready for production</h2>\n";
    echo "<p>The following errors must be fixed:</p>\n";
    echo "<ul>\n";
    foreach ($errors as $error) {
        echo "<li>$error</li>\n";
    }
    echo "</ul>\n";
}
?>
