<?php
/**
 * Broadcast Executor Script
 * Runs in background to execute broadcasts
 */

// Enable error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Set timezone
date_default_timezone_set('Asia/Jakarta');

// Load environment variables
if (file_exists('../../.env')) {
    $lines = file('../../.env', FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    foreach ($lines as $line) {
        if (strpos($line, '=') !== false && strpos($line, '#') !== 0) {
            list($key, $value) = explode('=', $line, 2);
            $_ENV[trim($key)] = trim($value);
        }
    }
}

// Load dependencies
require_once '../../config/database.php';
require_once '../../classes/DatabaseLogger.php';
require_once '../includes/BroadcastManager.php';

// Get broadcast ID from command line argument
$broadcastId = $argv[1] ?? null;

if (!$broadcastId) {
    error_log("Broadcast Executor: No broadcast ID provided");
    exit(1);
}

try {
    // Initialize components
    $database = new Database();
    $logger = new DatabaseLogger($database->getConnection(), 'broadcast_executor');
    $broadcastManager = new BroadcastManager($database, $logger);
    
    $logger->log('INFO', 'Broadcast executor started', ['broadcast_id' => $broadcastId]);
    
    // Get broadcast details
    $pdo = $database->getConnection();
    $stmt = $pdo->prepare("
        SELECT bl.*, au.username as admin_username
        FROM broadcast_logs bl
        LEFT JOIN admin_users au ON bl.admin_id = au.id
        WHERE bl.id = ? AND bl.status = 'running'
    ");
    $stmt->execute([$broadcastId]);
    $broadcast = $stmt->fetch();
    
    if (!$broadcast) {
        $logger->log('WARNING', 'Broadcast not found or not running', ['broadcast_id' => $broadcastId]);
        exit(1);
    }
    
    // Get target users
    $users = getTargetUsers($pdo, $broadcast);
    
    if (empty($users)) {
        $logger->log('WARNING', 'No target users found', ['broadcast_id' => $broadcastId]);
        updateBroadcastStatus($pdo, $broadcastId, 'completed', 0, 0);
        exit(0);
    }
    
    $totalUsers = count($users);
    $successCount = 0;
    $failedCount = 0;
    
    // Update total users count
    $stmt = $pdo->prepare("UPDATE broadcast_logs SET total_users = ? WHERE id = ?");
    $stmt->execute([$totalUsers, $broadcastId]);
    
    $logger->log('INFO', 'Starting broadcast execution', [
        'broadcast_id' => $broadcastId,
        'total_users' => $totalUsers
    ]);
    
    // Process each user
    foreach ($users as $user) {
        try {
            // Check for cancellation
            if (file_exists('../../broadcast_cancelled.flag')) {
                $logger->log('INFO', 'Broadcast cancelled by admin', ['broadcast_id' => $broadcastId]);
                updateBroadcastStatus($pdo, $broadcastId, 'cancelled', $successCount, $failedCount);
                unlink('../../broadcast_cancelled.flag');
                exit(0);
            }
            
            // Send message to user
            $result = sendMessageToUser($user['id'], $broadcast['message']);
            
            if ($result['success']) {
                $successCount++;
                $status = 'delivered';
            } else {
                $failedCount++;
                $status = 'failed';
            }
            
            // Log recipient
            $stmt = $pdo->prepare("
                INSERT INTO broadcast_recipients 
                (broadcast_id, user_id, status, error_message, created_at) 
                VALUES (?, ?, ?, ?, NOW())
            ");
            $stmt->execute([
                $broadcastId,
                $user['id'],
                $status,
                $result['error'] ?? null
            ]);
            
            // Update broadcast progress
            $stmt = $pdo->prepare("
                UPDATE broadcast_logs 
                SET success_count = ?, failed_count = ? 
                WHERE id = ?
            ");
            $stmt->execute([$successCount, $failedCount, $broadcastId]);
            
            // Rate limiting - wait between messages
            usleep(100000); // 0.1 second delay
            
        } catch (Exception $e) {
            $logger->log('ERROR', 'Failed to send message to user', [
                'broadcast_id' => $broadcastId,
                'user_id' => $user['id'],
                'error' => $e->getMessage()
            ]);
            $failedCount++;
        }
    }
    
    // Mark broadcast as completed
    updateBroadcastStatus($pdo, $broadcastId, 'completed', $successCount, $failedCount);
    
    $logger->log('INFO', 'Broadcast execution completed', [
        'broadcast_id' => $broadcastId,
        'success_count' => $successCount,
        'failed_count' => $failedCount
    ]);
    
} catch (Exception $e) {
    $logger->log('ERROR', 'Broadcast executor failed', [
        'broadcast_id' => $broadcastId,
        'error' => $e->getMessage()
    ]);
    
    // Mark broadcast as failed
    if (isset($pdo)) {
        updateBroadcastStatus($pdo, $broadcastId, 'failed', 0, 0);
    }
    
    exit(1);
}

/**
 * Get target users based on broadcast settings
 */
function getTargetUsers($pdo, $broadcast) {
    switch ($broadcast['target_type']) {
        case 'all':
            $stmt = $pdo->query("SELECT id, username FROM users ORDER BY id");
            break;
            
        case 'active':
            $stmt = $pdo->prepare("
                SELECT DISTINCT u.id, u.username 
                FROM users u
                INNER JOIN bot_logs bl ON u.id = bl.user_id
                WHERE bl.created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)
                ORDER BY u.id
            ");
            $stmt->execute();
            break;
            
        case 'specific':
            $userIds = json_decode($broadcast['target_users'], true);
            if (empty($userIds)) {
                return [];
            }
            $placeholders = str_repeat('?,', count($userIds) - 1) . '?';
            $stmt = $pdo->prepare("
                SELECT id, username FROM users 
                WHERE id IN ({$placeholders})
            ");
            $stmt->execute($userIds);
            break;
            
        default:
            return [];
    }
    
    return $stmt->fetchAll();
}

/**
 * Send message to user via Telegram API
 */
function sendMessageToUser($userId, $message) {
    try {
        $config = require '../../config.php';
        $botToken = $config['bot_token'];
        
        $url = "https://api.telegram.org/bot{$botToken}/sendMessage";
        $data = [
            'chat_id' => $userId,
            'text' => $message,
            'parse_mode' => 'HTML'
        ];
        
        $options = [
            'http' => [
                'header' => "Content-type: application/x-www-form-urlencoded\r\n",
                'method' => 'POST',
                'content' => http_build_query($data)
            ]
        ];
        
        $context = stream_context_create($options);
        $response = file_get_contents($url, false, $context);
        $result = json_decode($response, true);
        
        if ($result['ok']) {
            return ['success' => true];
        } else {
            return [
                'success' => false,
                'error' => $result['description'] ?? 'Unknown error'
            ];
        }
        
    } catch (Exception $e) {
        return [
            'success' => false,
            'error' => $e->getMessage()
        ];
    }
}

/**
 * Update broadcast status
 */
function updateBroadcastStatus($pdo, $broadcastId, $status, $successCount, $failedCount) {
    $stmt = $pdo->prepare("
        UPDATE broadcast_logs 
        SET status = ?, 
            success_count = ?, 
            failed_count = ?,
            completed_at = NOW()
        WHERE id = ?
    ");
    $stmt->execute([$status, $successCount, $failedCount, $broadcastId]);
}
