<?php
/**
 * System Verification Script
 * Check all critical files for errors
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

echo "<h1>🔍 System Verification</h1>\n";
echo "<p>Checking all critical files for errors...</p>\n";

$criticalFiles = [
    // Config files
    '../config/database.php',
    '../config/app_config.php',
    '../config/hosting_config.php',
    
    // Admin files
    'index.php',
    'includes/AdminController.php',
    'includes/BotManager.php',
    'includes/BroadcastManager.php',
    'includes/AuthManager.php',
    
    // Template files
    'templates/dashboard.php',
    'templates/bot_management.php',
    'templates/broadcast.php',
    'templates/users.php',
    'templates/analytics.php',
    'templates/logs.php',
    'templates/settings.php',
    
    // Core files
    '../telegram_bot.php',
    '../classes/TelegramAPI.php',
    '../classes/DatabaseLogger.php',
    '../classes/MovieScraper.php',
    '../classes/AdultScraper.php',
];

$results = [];
$errorCount = 0;

foreach ($criticalFiles as $file) {
    $status = 'success';
    $message = '';
    
    if (file_exists($file)) {
        // Check syntax
        $output = [];
        $return = 0;
        exec("php -l \"$file\" 2>&1", $output, $return);
        
        if ($return === 0) {
            $status = 'success';
            $message = 'Syntax OK';
        } else {
            $status = 'error';
            $message = implode("\n", $output);
            $errorCount++;
        }
    } else {
        $status = 'warning';
        $message = 'File not found';
    }
    
    $results[] = [
        'file' => $file,
        'status' => $status,
        'message' => $message
    ];
}

// Display results
echo "<h2>Verification Results</h2>\n";
echo "<table border='1' style='border-collapse: collapse; width: 100%;'>\n";
echo "<tr><th>File</th><th>Status</th><th>Message</th></tr>\n";

foreach ($results as $result) {
    $color = $result['status'] === 'success' ? 'green' : ($result['status'] === 'error' ? 'red' : 'orange');
    $icon = $result['status'] === 'success' ? '✅' : ($result['status'] === 'error' ? '❌' : '⚠️');
    
    echo "<tr>";
    echo "<td>{$result['file']}</td>";
    echo "<td style='color: $color;'>$icon {$result['status']}</td>";
    echo "<td><pre style='font-size: 10px;'>" . htmlspecialchars($result['message']) . "</pre></td>";
    echo "</tr>\n";
}

echo "</table>\n";

// Summary
echo "<br><h2>Summary</h2>\n";
$successCount = count(array_filter($results, function($r) { return $r['status'] === 'success'; }));
$warningCount = count(array_filter($results, function($r) { return $r['status'] === 'warning'; }));
$errorCount_final = count(array_filter($results, function($r) { return $r['status'] === 'error'; }));

echo "<p>Total files checked: " . count($results) . "</p>\n";
echo "<p>✅ Successful: $successCount</p>\n";
echo "<p>⚠️ Warnings: $warningCount</p>\n";
echo "<p>❌ Errors: $errorCount_final</p>\n";

if ($errorCount_final === 0) {
    echo "<br><h2 style='color: green;'>✅ All files verified successfully!</h2>\n";
    echo "<p>The system is ready for production.</p>\n";
} else {
    echo "<br><h2 style='color: red;'>❌ Errors found!</h2>\n";
    echo "<p>Please fix the errors before deploying to production.</p>\n";
}

// Test database connection
echo "<br><h2>Database Connection Test</h2>\n";
try {
    require_once '../config/database.php';
    $database = new Database();
    $pdo = $database->getConnection();
    echo "✅ Database connection successful<br>\n";
    
    // Test bot settings
    $stmt = $pdo->query("SELECT COUNT(*) FROM bot_settings");
    $count = $stmt->fetchColumn();
    echo "✅ Bot settings table: $count configurations<br>\n";
    
} catch (Exception $e) {
    echo "❌ Database connection failed: " . $e->getMessage() . "<br>\n";
}

// Test bot token
echo "<br><h2>Bot Token Test</h2>\n";
try {
    $stmt = $pdo->query("SELECT setting_value FROM bot_settings WHERE setting_name = 'bot_token'");
    $botToken = $stmt->fetchColumn();
    
    if ($botToken) {
        echo "✅ Bot token found: " . substr($botToken, 0, 10) . "...<br>\n";
        
        $url = "https://api.telegram.org/bot{$botToken}/getMe";
        $context = stream_context_create([
            'http' => [
                'timeout' => 10,
                'method' => 'GET',
                'header' => 'User-Agent: AdminPanel/1.0',
                'ignore_errors' => true
            ]
        ]);
        
        $response = file_get_contents($url, false, $context);
        if ($response !== false) {
            $data = json_decode($response, true);
            if ($data && $data['ok']) {
                echo "✅ Bot token is valid! Bot: @" . $data['result']['username'] . "<br>\n";
            } else {
                echo "❌ Bot token is invalid<br>\n";
            }
        }
    } else {
        echo "❌ Bot token not found in database<br>\n";
    }
} catch (Exception $e) {
    echo "❌ Error testing bot token: " . $e->getMessage() . "<br>\n";
}
?>
