<?php
/**
 * Logs API Endpoint
 * Handles log retrieval, clearing, and settings management
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Load dependencies
require_once '../config/database.php';
require_once '../classes/DatabaseLogger.php';

try {
    $database = new Database();
    $logger = new DatabaseLogger($database->getConnection(), 'api');
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Database connection failed']);
    exit;
}

$method = $_SERVER['REQUEST_METHOD'];

switch ($method) {
    case 'GET':
        handleGetLogs($logger);
        break;
    case 'POST':
        handleUpdateSettings($logger);
        break;
    case 'DELETE':
        handleClearLogs($logger);
        break;
    default:
        http_response_code(405);
        echo json_encode(['success' => false, 'message' => 'Method not allowed']);
}

function handleGetLogs($logger) {
    $type = $_GET['type'] ?? 'bot';
    $limit = (int)($_GET['limit'] ?? 50);
    $offset = (int)($_GET['offset'] ?? 0);
    
    try {
        if ($type === 'bot') {
            $logType = $_GET['log_type'] ?? null;
            $logs = $logger->getLogs($logType, $limit, $offset);
        } else {
            $level = $_GET['level'] ?? null;
            $component = $_GET['component'] ?? null;
            $logs = $logger->getSystemLogs($level, $component, $limit, $offset);
        }
        
        echo json_encode([
            'success' => true,
            'logs' => $logs,
            'count' => count($logs)
        ]);
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Error retrieving logs: ' . $e->getMessage()
        ]);
    }
}

function handleUpdateSettings($logger) {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Invalid JSON data']);
        return;
    }
    
    try {
        $updated = 0;
        
        if (isset($input['auto_reset_hours'])) {
            $logger->updateLogSetting('auto_reset_hours', $input['auto_reset_hours']);
            $updated++;
        }
        
        if (isset($input['max_log_entries'])) {
            $logger->updateLogSetting('max_log_entries', $input['max_log_entries']);
            $updated++;
        }
        
        if (isset($input['retention_days'])) {
            $logger->updateLogSetting('log_retention_days', $input['retention_days']);
            $updated++;
        }
        
        if (isset($input['enable_debug_logs'])) {
            $logger->updateLogSetting('enable_debug_logs', $input['enable_debug_logs']);
            $updated++;
        }
        
        if (isset($input['log_level'])) {
            $logger->updateLogSetting('log_level', $input['log_level']);
            $updated++;
        }
        
        echo json_encode([
            'success' => true,
            'message' => "Updated {$updated} settings",
            'updated' => $updated
        ]);
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Error updating settings: ' . $e->getMessage()
        ]);
    }
}

function handleClearLogs($logger) {
    try {
        $type = $_GET['type'] ?? null;
        $olderThan = $_GET['older_than'] ?? null;
        
        $deletedCount = $logger->clearLogs($type, $olderThan);
        
        echo json_encode([
            'success' => true,
            'message' => "Cleared {$deletedCount} log entries",
            'deleted' => $deletedCount
        ]);
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Error clearing logs: ' . $e->getMessage()
        ]);
    }
}

