<?php
/**
 * Adult Content Scraper Class
 * Handles adult content scraping with caching and error handling
 */

class AdultScraper {
    private $logger;
    private $cache;
    private $cacheTimeout = 1800; // 30 minutes
    private $apiUrl;
    
    public function __construct($logger) {
        $this->logger = $logger;
        $this->cache = [];
        
        // Load environment variables
        if (file_exists('.env')) {
            $lines = file('.env', FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
            foreach ($lines as $line) {
                if (strpos($line, '=') !== false && strpos($line, '#') !== 0) {
                    list($key, $value) = explode('=', $line, 2);
                    $_ENV[trim($key)] = trim($value);
                }
            }
        }
        
        // Get API URL from environment or use default
        $this->apiUrl = $_ENV['ADULT_API_URL'] ?? 'https://tes.apivalidasi.my.id/api_adult.php';
        
        // Log API URL configuration
        $this->logger->info("AdultScraper initialized", [
            'api_url' => $this->apiUrl,
            'cache_timeout' => $this->cacheTimeout
        ]);
    }
    
    /**
     * Get current API URL
     */
    public function getApiUrl() {
        return $this->apiUrl;
    }
    
    /**
     * Set API URL (for testing or dynamic configuration)
     */
    public function setApiUrl($url) {
        $this->apiUrl = $url;
        $this->logger->info("API URL updated", ['new_url' => $url]);
    }
    
    /**
     * Scrape adult content with caching
     */
    public function scrapeAdultContent($query) {
        try {
            $cacheKey = 'adult_' . md5($query);
            
            // Check cache first
            if (isset($this->cache[$cacheKey])) {
                $cached = $this->cache[$cacheKey];
                if (time() - $cached['timestamp'] < $this->cacheTimeout) {
                    $this->logger->info("Returning cached adult content", ['query' => $query]);
                    return $cached['data'];
                }
            }
            
            $searchUrl = $this->apiUrl . '?search=' . urlencode($query);
            
            $this->logger->info("Making API request", [
                'url' => $searchUrl,
                'query' => $query
            ]);
            
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $searchUrl);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
            curl_setopt($ch, CURLOPT_TIMEOUT, 30);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $error = curl_error($ch);
            curl_close($ch);
            
            if ($error) {
                throw new Exception("cURL Error: {$error}");
            }
            
            if ($httpCode !== 200) {
                throw new Exception("HTTP Error: {$httpCode}");
            }
            
            if (!$response) {
                throw new Exception("Empty response");
            }
            
            // Parse JSON response
            $data = json_decode($response, true);
            
            if (!$data || !isset($data['data']) || !is_array($data['data'])) {
                throw new Exception("Invalid JSON response");
            }
            
            $adultContent = [];
            $items = $data['data'];
            $count = 0;
            $maxResults = 5;
            
            foreach ($items as $item) {
                if ($count >= $maxResults) break;
                
                $content = [];
                $content['id'] = isset($item['id']) ? $item['id'] : '';
                $content['title'] = isset($item['title']) ? $item['title'] : '';
                $content['link'] = isset($item['link']) ? $item['link'] : '';
                $content['image_url'] = isset($item['image_url']) ? $item['image_url'] : '';
                $content['genre'] = isset($item['Genre']) ? $item['Genre'] : '';
                $content['quality'] = isset($item['Kualitas']) ? $item['Kualitas'] : '';
                $content['year'] = isset($item['Tahun']) ? $item['Tahun'] : '';
                $content['duration'] = isset($item['Durasi']) ? $item['Durasi'] : '';
                $content['embed_url'] = isset($item['embed_url']) ? $item['embed_url'] : '';
                
                if (!empty($content['title'])) {
                    $adultContent[] = $content;
                    $count++;
                }
            }
            
            // Cache the results
            $this->cache[$cacheKey] = [
                'data' => $adultContent,
                'timestamp' => time()
            ];
            
            $this->logger->info("Adult content scraped successfully", [
                'query' => $query,
                'count' => count($adultContent)
            ]);
            
            return $adultContent;
            
        } catch (Exception $e) {
            $this->logger->error("Failed to scrape adult content", [
                'query' => $query,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }
    
    /**
     * Get random adult content
     */
    public function getRandomAdultContent() {
        try {
            $cacheKey = 'adult_random';
            
            // Check cache first
            if (isset($this->cache[$cacheKey])) {
                $cached = $this->cache[$cacheKey];
                if (time() - $cached['timestamp'] < $this->cacheTimeout) {
                    $this->logger->info("Returning cached random adult content");
                    return $cached['data'];
                }
            }
            
            $apiUrl = $this->apiUrl . '?random=1';
            
            $this->logger->info("Making random API request", [
                'url' => $apiUrl
            ]);
            
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $apiUrl);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
            curl_setopt($ch, CURLOPT_TIMEOUT, 30);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $error = curl_error($ch);
            curl_close($ch);
            
            if ($error) {
                throw new Exception("cURL Error: {$error}");
            }
            
            if ($httpCode !== 200) {
                throw new Exception("HTTP Error: {$httpCode}");
            }
            
            if (!$response) {
                throw new Exception("Empty response");
            }
            
            // Parse JSON response
            $data = json_decode($response, true);
            
            if (!$data || !isset($data['data']) || !is_array($data['data'])) {
                throw new Exception("Invalid JSON response");
            }
            
            // Cache the results
            $this->cache[$cacheKey] = [
                'data' => $data['data'],
                'timestamp' => time()
            ];
            
            $this->logger->info("Random adult content retrieved successfully", [
                'count' => count($data['data'])
            ]);
            
            return $data['data'];
            
        } catch (Exception $e) {
            $this->logger->error("Failed to get random adult content", [
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }
    
    /**
     * Get adult content detail by ID
     */
    public function getAdultContentDetail($contentId) {
        try {
            $cacheKey = 'adult_detail_' . md5($contentId);
            
            // Check cache first
            if (isset($this->cache[$cacheKey])) {
                $cached = $this->cache[$cacheKey];
                if (time() - $cached['timestamp'] < $this->cacheTimeout) {
                    $this->logger->info("Returning cached adult content detail", ['id' => $contentId]);
                    return $cached['data'];
                }
            }
            
            $detailUrl = $this->apiUrl . '?id=' . urlencode($contentId);
            
            $this->logger->info("Making detail API request", [
                'url' => $detailUrl,
                'content_id' => $contentId
            ]);
            
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $detailUrl);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
            curl_setopt($ch, CURLOPT_TIMEOUT, 30);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $error = curl_error($ch);
            curl_close($ch);
            
            if ($error) {
                throw new Exception("cURL Error: {$error}");
            }
            
            if ($httpCode !== 200) {
                throw new Exception("HTTP Error: {$httpCode}");
            }
            
            if (!$response) {
                throw new Exception("Empty response");
            }
            
            // Parse JSON response
            $data = json_decode($response, true);
            
            if (!$data || !isset($data['data'])) {
                throw new Exception("Invalid JSON response");
            }
            
            $detail = $data['data'];
            $content = [];
            $content['id'] = isset($detail['id']) ? $detail['id'] : '';
            $content['title'] = isset($detail['title']) ? $detail['title'] : '';
            $content['link'] = isset($detail['link']) ? $detail['link'] : '';
            $content['image_url'] = isset($detail['image_url']) ? $detail['image_url'] : '';
            $content['genre'] = isset($detail['Genre']) ? $detail['Genre'] : '';
            $content['quality'] = isset($detail['Kualitas']) ? $detail['Kualitas'] : '';
            $content['year'] = isset($detail['Tahun']) ? $detail['Tahun'] : '';
            $content['duration'] = isset($detail['Durasi']) ? $detail['Durasi'] : '';
            $content['embed_url'] = isset($detail['embed_url']) ? $detail['embed_url'] : '';
            
            // Cache the result
            $this->cache[$cacheKey] = [
                'data' => $content,
                'timestamp' => time()
            ];
            
            $this->logger->info("Adult content detail retrieved successfully", ['id' => $contentId]);
            return $content;
            
        } catch (Exception $e) {
            $this->logger->error("Failed to get adult content detail", [
                'id' => $contentId,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }
    
    /**
     * Get total adult content count
     */
    public function getTotalAdultContent() {
        try {
            $cacheKey = 'total_adult';
            
            // Check cache first
            if (isset($this->cache[$cacheKey])) {
                $cached = $this->cache[$cacheKey];
                if (time() - $cached['timestamp'] < $this->cacheTimeout) {
                    return $cached['data'];
                }
            }
            
            $searchUrl = $this->apiUrl . '?search=';
            
            $this->logger->info("Making total count API request", [
                'url' => $searchUrl
            ]);
            
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $searchUrl);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
            curl_setopt($ch, CURLOPT_TIMEOUT, 30);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            
            if ($httpCode !== 200 || !$response) {
                return 0;
            }
            
            // Parse JSON response
            $data = json_decode($response, true);
            
            if (!$data) {
                return 0;
            }
            
            $totalAdult = 0;
            
            // Check for total_data field
            if (isset($data['total_data'])) {
                $totalAdult = (int)$data['total_data'];
            } elseif (isset($data['data']) && is_array($data['data'])) {
                $totalAdult = count($data['data']);
            }
            
            // Cache the result
            $this->cache[$cacheKey] = [
                'data' => $totalAdult,
                'timestamp' => time()
            ];
            
            $this->logger->info("Total adult content calculated", ['total' => $totalAdult]);
            return $totalAdult;
            
        } catch (Exception $e) {
            $this->logger->error("Failed to get total adult content", ['error' => $e->getMessage()]);
            return 0;
        }
    }
}


