<?php
/**
 * Broadcast Management Class
 * Handles broadcast operations with async processing
 */

class BroadcastManager {
    private $db;
    private $logger;
    private $security;
    private $telegram;
    
    public function __construct($database, $logger, $security, $telegram) {
        $this->db = $database;
        $this->logger = $logger;
        $this->security = $security;
        $this->telegram = $telegram;
    }
    
    /**
     * Create broadcast job
     */
    public function createBroadcast($adminId, $message) {
        try {
            $pdo = $this->db->getConnection();
            
            // Validate message
            $validation = $this->security->validateBroadcastMessage($message);
            if (!$validation['valid']) {
                return [
                    'success' => false,
                    'message' => $validation['message']
                ];
            }
            
            $cleanMessage = $validation['cleaned_message'];
            
            // Get total users
            $userManager = new UserManager($this->db, $this->logger, $this->security);
            $totalUsers = $userManager->getUserCount();
            
            if ($totalUsers === 0) {
                return [
                    'success' => false,
                    'message' => '❌ Tidak ada user untuk broadcast!'
                ];
            }
            
            // Create broadcast record
            $stmt = $pdo->prepare("
                INSERT INTO broadcast_logs (admin_id, message, total_users, status) 
                VALUES (?, ?, ?, 'pending')
            ");
            $stmt->execute([$adminId, $cleanMessage, $totalUsers]);
            $broadcastId = $pdo->lastInsertId();
            
            $this->logger->info("Broadcast created", [
                'broadcast_id' => $broadcastId,
                'admin_id' => $adminId,
                'total_users' => $totalUsers
            ]);
            
            return [
                'success' => true,
                'broadcast_id' => $broadcastId,
                'total_users' => $totalUsers,
                'message' => "📢 <b>Broadcast Preview:</b>\n\n" .
                           "📝 <b>Pesan:</b> " . substr($cleanMessage, 0, 100) . 
                           (strlen($cleanMessage) > 100 ? "..." : "") . "\n" .
                           "👥 <b>Target:</b> {$totalUsers} user\n\n" .
                           "⚠️ <b>Apakah Anda yakin ingin mengirim broadcast ini?</b>\n\n" .
                           "💡 <b>Ketik /broadcast_confirm untuk konfirmasi</b>"
            ];
            
        } catch (Exception $e) {
            $this->logger->error("Failed to create broadcast", [
                'admin_id' => $adminId,
                'error' => $e->getMessage()
            ]);
            return [
                'success' => false,
                'message' => '❌ Gagal membuat broadcast!'
            ];
        }
    }
    
    /**
     * Execute broadcast with async processing
     */
    public function executeBroadcast($broadcastId, $adminChatId) {
        try {
            $pdo = $this->db->getConnection();
            
            // Get broadcast details
            $stmt = $pdo->prepare("SELECT * FROM broadcast_logs WHERE id = ? AND status = 'pending'");
            $stmt->execute([$broadcastId]);
            $broadcast = $stmt->fetch();
            
            if (!$broadcast) {
                return [
                    'success' => false,
                    'message' => '❌ Broadcast tidak ditemukan atau sudah diproses!'
                ];
            }
            
            // Update status to running
            $stmt = $pdo->prepare("UPDATE broadcast_logs SET status = 'running' WHERE id = ?");
            $stmt->execute([$broadcastId]);
            
            // Start async broadcast process
            $this->startAsyncBroadcast($broadcastId, $adminChatId);
            
            return [
                'success' => true,
                'message' => '🔄 <b>Broadcast dimulai...</b>\n\n📊 <b>Progress akan ditampilkan secara real-time</b>'
            ];
            
        } catch (Exception $e) {
            $this->logger->error("Failed to execute broadcast", [
                'broadcast_id' => $broadcastId,
                'error' => $e->getMessage()
            ]);
            return [
                'success' => false,
                'message' => '❌ Gagal menjalankan broadcast!'
            ];
        }
    }
    
    /**
     * Start async broadcast process
     */
    private function startAsyncBroadcast($broadcastId, $adminChatId) {
        // Create a background process
        $script = __DIR__ . '/../scripts/async_broadcast.php';
        $command = "php {$script} {$broadcastId {$adminChatId} > /dev/null 2>&1 &";
        
        // Execute in background
        exec($command);
        
        $this->logger->info("Async broadcast started", [
            'broadcast_id' => $broadcastId,
            'admin_chat_id' => $adminChatId
        ]);
    }
    
    /**
     * Cancel broadcast
     */
    public function cancelBroadcast($broadcastId, $adminId) {
        try {
            $pdo = $this->db->getConnection();
            
            $stmt = $pdo->prepare("
                UPDATE broadcast_logs 
                SET status = 'cancelled' 
                WHERE id = ? AND admin_id = ? AND status IN ('pending', 'running')
            ");
            $result = $stmt->execute([$broadcastId, $adminId]);
            
            if ($result && $stmt->rowCount() > 0) {
                $this->logger->info("Broadcast cancelled", [
                    'broadcast_id' => $broadcastId,
                    'admin_id' => $adminId
                ]);
                return true;
            }
            
            return false;
            
        } catch (Exception $e) {
            $this->logger->error("Failed to cancel broadcast", [
                'broadcast_id' => $broadcastId,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }
    
    /**
     * Get broadcast status
     */
    public function getBroadcastStatus($broadcastId) {
        try {
            $pdo = $this->db->getConnection();
            $stmt = $pdo->prepare("SELECT * FROM broadcast_logs WHERE id = ?");
            $stmt->execute([$broadcastId]);
            return $stmt->fetch();
        } catch (Exception $e) {
            $this->logger->error("Failed to get broadcast status", [
                'broadcast_id' => $broadcastId,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }
    
    /**
     * Get broadcast history
     */
    public function getBroadcastHistory($adminId = null, $limit = 10) {
        try {
            $pdo = $this->db->getConnection();
            
            if ($adminId) {
                $stmt = $pdo->prepare("
                    SELECT * FROM broadcast_logs 
                    WHERE admin_id = ? 
                    ORDER BY created_at DESC 
                    LIMIT ?
                ");
                $stmt->execute([$adminId, $limit]);
            } else {
                $stmt = $pdo->prepare("
                    SELECT * FROM broadcast_logs 
                    ORDER BY created_at DESC 
                    LIMIT ?
                ");
                $stmt->execute([$limit]);
            }
            
            return $stmt->fetchAll();
        } catch (Exception $e) {
            $this->logger->error("Failed to get broadcast history", [
                'admin_id' => $adminId,
                'error' => $e->getMessage()
            ]);
            return [];
        }
    }
    
    /**
     * Update broadcast progress
     */
    public function updateBroadcastProgress($broadcastId, $successCount, $failedCount, $status = 'running') {
        try {
            $pdo = $this->db->getConnection();
            
            $stmt = $pdo->prepare("
                UPDATE broadcast_logs 
                SET success_count = ?, failed_count = ?, status = ?
                WHERE id = ?
            ");
            
            $result = $stmt->execute([$successCount, $failedCount, $status, $broadcastId]);
            
            if ($result && $status === 'completed') {
                // Calculate duration
                $stmt = $pdo->prepare("
                    UPDATE broadcast_logs 
                    SET completed_at = NOW(), 
                        duration = TIMESTAMPDIFF(SECOND, created_at, NOW())
                    WHERE id = ?
                ");
                $stmt->execute([$broadcastId]);
            }
            
            return $result;
            
        } catch (Exception $e) {
            $this->logger->error("Failed to update broadcast progress", [
                'broadcast_id' => $broadcastId,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }
}



