<?php

class DatabaseLogger {
    private $pdo;
    private $component;
    
    public function __construct($pdo, $component = 'system') {
        $this->pdo = $pdo;
        $this->component = $component;
    }
    
    public function log($level, $message, $data = null, $userId = null) {
        try {
            $stmt = $this->pdo->prepare("
                INSERT INTO system_logs (level, component, message, user_id, data, ip_address, user_agent) 
                VALUES (?, ?, ?, ?, ?, ?, ?)
            ");
            
            $stmt->execute([
                $level,
                $this->component,
                $message,
                $userId,
                $data ? json_encode($data) : null,
                $this->getClientIP(),
                $_SERVER['HTTP_USER_AGENT'] ?? null
            ]);
        } catch (Exception $e) {
            // Fallback to file logging if database fails
            error_log("Database logging failed: " . $e->getMessage());
        }
    }
    
    public function logBot($type, $message, $userId = null, $username = null, $action = null, $details = null) {
        try {
            $stmt = $this->pdo->prepare("
                INSERT INTO bot_logs (log_type, user_id, username, action, details, ip_address, user_agent) 
                VALUES (?, ?, ?, ?, ?, ?, ?)
            ");
            
            $stmt->execute([
                $type,
                $userId,
                $username,
                $action,
                $details,
                $this->getClientIP(),
                $_SERVER['HTTP_USER_AGENT'] ?? null
            ]);
        } catch (Exception $e) {
            // Fallback to file logging
            $this->logToFile($type, $message, $userId, $username);
        }
    }
    
    public function getLogs($type = null, $limit = 100, $offset = 0) {
        try {
            $sql = "SELECT * FROM bot_logs";
            $params = [];
            
            if ($type) {
                $sql .= " WHERE log_type = ?";
                $params[] = $type;
            }
            
            $sql .= " ORDER BY created_at DESC LIMIT ? OFFSET ?";
            $params[] = $limit;
            $params[] = $offset;
            
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            return [];
        }
    }
    
    public function getSystemLogs($level = null, $component = null, $limit = 100, $offset = 0) {
        try {
            $sql = "SELECT * FROM system_logs WHERE 1=1";
            $params = [];
            
            if ($level) {
                $sql .= " AND level = ?";
                $params[] = $level;
            }
            
            if ($component) {
                $sql .= " AND component = ?";
                $params[] = $component;
            }
            
            $sql .= " ORDER BY created_at DESC LIMIT ? OFFSET ?";
            $params[] = $limit;
            $params[] = $offset;
            
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            return [];
        }
    }
    
    public function clearLogs($type = null, $olderThan = null) {
        try {
            if ($type) {
                $sql = "DELETE FROM bot_logs WHERE log_type = ?";
                $params = [$type];
                
                if ($olderThan) {
                    $sql .= " AND created_at < ?";
                    $params[] = $olderThan;
                }
            } else {
                $sql = "DELETE FROM bot_logs";
                $params = [];
                
                if ($olderThan) {
                    $sql .= " WHERE created_at < ?";
                    $params[] = $olderThan;
                }
            }
            
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            
            return $stmt->rowCount();
        } catch (Exception $e) {
            return 0;
        }
    }
    
    public function getLogSettings() {
        try {
            $stmt = $this->pdo->prepare("SELECT * FROM log_settings");
            $stmt->execute();
            
            $settings = [];
            while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                $settings[$row['setting_name']] = $row['setting_value'];
            }
            
            return $settings;
        } catch (Exception $e) {
            return [];
        }
    }
    
    public function updateLogSetting($name, $value) {
        try {
            $stmt = $this->pdo->prepare("
                UPDATE log_settings SET setting_value = ? WHERE setting_name = ?
            ");
            return $stmt->execute([$value, $name]);
        } catch (Exception $e) {
            return false;
        }
    }
    
    public function autoCleanup() {
        try {
            $settings = $this->getLogSettings();
            $retentionDays = (int)($settings['log_retention_days'] ?? 30);
            $maxEntries = (int)($settings['max_log_entries'] ?? 10000);
            
            // Clean old logs
            if ($retentionDays > 0) {
                $cutoffDate = date('Y-m-d H:i:s', strtotime("-{$retentionDays} days"));
                $this->clearLogs(null, $cutoffDate);
            }
            
            // Limit total entries
            $stmt = $this->pdo->prepare("SELECT COUNT(*) FROM bot_logs");
            $stmt->execute();
            $totalLogs = $stmt->fetchColumn();
            
            if ($totalLogs > $maxEntries) {
                $deleteCount = $totalLogs - $maxEntries;
                $stmt = $this->pdo->prepare("
                    DELETE FROM bot_logs ORDER BY created_at ASC LIMIT ?
                ");
                $stmt->execute([$deleteCount]);
            }
            
            return true;
        } catch (Exception $e) {
            return false;
        }
    }
    
    private function getClientIP() {
        $ipKeys = ['HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'REMOTE_ADDR'];
        foreach ($ipKeys as $key) {
            if (!empty($_SERVER[$key])) {
                $ip = $_SERVER[$key];
                if (strpos($ip, ',') !== false) {
                    $ip = trim(explode(',', $ip)[0]);
                }
                if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
                    return $ip;
                }
            }
        }
        return $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    }
    
    private function logToFile($type, $message, $userId = null, $username = null) {
        $timestamp = date('Y-m-d H:i:s');
        $logEntry = "[{$timestamp}] [{$type}]";
        
        if ($userId) {
            $logEntry .= " User: {$userId}";
        }
        if ($username) {
            $logEntry .= " (@{$username})";
        }
        
        $logEntry .= " - {$message}\n";
        
        file_put_contents('bot_log.txt', $logEntry, FILE_APPEND | LOCK_EX);
    }
}

