<?php
/**
 * Telegram API Class
 * Handles all Telegram API interactions with proper error handling
 */

class TelegramAPI {
    private $botToken;
    private $logger;
    private $rateLimiter;
    
    public function __construct($botToken, $logger) {
        $this->botToken = $botToken;
        $this->logger = $logger;
        $this->rateLimiter = [];
    }
    
    /**
     * Send message to Telegram
     */
    public function sendMessage($chatId, $text, $parseMode = 'HTML', $replyToMessageId = null, $disableWebPagePreview = false) {
        try {
            $url = "https://api.telegram.org/bot{$this->botToken}/sendMessage";
            $data = [
                'chat_id' => $chatId,
                'text' => $text,
                'parse_mode' => $parseMode,
                'disable_web_page_preview' => $disableWebPagePreview
            ];
            
            if ($replyToMessageId) {
                $data['reply_to_message_id'] = $replyToMessageId;
            }
            
            $result = $this->makeRequest($url, $data);
            
            if ($result['success']) {
                $this->logger->info("Message sent successfully", [
                    'chat_id' => $chatId,
                    'message_length' => strlen($text)
                ]);
            } else {
                $this->logger->error("Failed to send message", [
                    'chat_id' => $chatId,
                    'error' => $result['error']
                ]);
            }
            
            return $result;
            
        } catch (Exception $e) {
            $this->logger->error("Exception in sendMessage", [
                'chat_id' => $chatId,
                'error' => $e->getMessage()
            ]);
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Send message with inline keyboard
     */
    public function sendMessageWithButtons($chatId, $text, $buttons, $replyToMessageId = null) {
        try {
            $url = "https://api.telegram.org/bot{$this->botToken}/sendMessage";
            $data = [
                'chat_id' => $chatId,
                'text' => $text,
                'parse_mode' => 'HTML',
                'reply_markup' => json_encode([
                    'inline_keyboard' => $buttons
                ])
            ];
            
            if ($replyToMessageId) {
                $data['reply_to_message_id'] = $replyToMessageId;
            }
            
            $result = $this->makeRequest($url, $data);
            
            if ($result['success']) {
                $this->logger->info("Message with buttons sent successfully", [
                    'chat_id' => $chatId,
                    'buttons_count' => count($buttons)
                ]);
            } else {
                $this->logger->error("Failed to send message with buttons", [
                    'chat_id' => $chatId,
                    'error' => $result['error']
                ]);
            }
            
            return $result;
            
        } catch (Exception $e) {
            $this->logger->error("Exception in sendMessageWithButtons", [
                'chat_id' => $chatId,
                'error' => $e->getMessage()
            ]);
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Send photo with caption
     */
    public function sendPhoto($chatId, $photoUrl, $caption = '', $parseMode = 'HTML') {
        try {
            $url = "https://api.telegram.org/bot{$this->botToken}/sendPhoto";
            $data = [
                'chat_id' => $chatId,
                'photo' => $photoUrl,
                'caption' => $caption,
                'parse_mode' => $parseMode
            ];
            
            $result = $this->makeRequest($url, $data);
            
            if ($result['success']) {
                $this->logger->info("Photo sent successfully", [
                    'chat_id' => $chatId,
                    'photo_url' => $photoUrl
                ]);
            } else {
                $this->logger->error("Failed to send photo", [
                    'chat_id' => $chatId,
                    'error' => $result['error']
                ]);
            }
            
            return $result;
            
        } catch (Exception $e) {
            $this->logger->error("Exception in sendPhoto", [
                'chat_id' => $chatId,
                'error' => $e->getMessage()
            ]);
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Edit message text
     */
    public function editMessage($chatId, $messageId, $text, $parseMode = 'HTML') {
        try {
            $url = "https://api.telegram.org/bot{$this->botToken}/editMessageText";
            $data = [
                'chat_id' => $chatId,
                'message_id' => $messageId,
                'text' => $text,
                'parse_mode' => $parseMode
            ];
            
            $result = $this->makeRequest($url, $data);
            
            if ($result['success']) {
                $this->logger->info("Message edited successfully", [
                    'chat_id' => $chatId,
                    'message_id' => $messageId
                ]);
            } else {
                $this->logger->error("Failed to edit message", [
                    'chat_id' => $chatId,
                    'message_id' => $messageId,
                    'error' => $result['error']
                ]);
            }
            
            return $result;
            
        } catch (Exception $e) {
            $this->logger->error("Exception in editMessage", [
                'chat_id' => $chatId,
                'message_id' => $messageId,
                'error' => $e->getMessage()
            ]);
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Delete message
     */
    public function deleteMessage($chatId, $messageId) {
        try {
            $url = "https://api.telegram.org/bot{$this->botToken}/deleteMessage";
            $data = [
                'chat_id' => $chatId,
                'message_id' => $messageId
            ];
            
            $result = $this->makeRequest($url, $data);
            
            if ($result['success']) {
                $this->logger->info("Message deleted successfully", [
                    'chat_id' => $chatId,
                    'message_id' => $messageId
                ]);
            } else {
                $this->logger->error("Failed to delete message", [
                    'chat_id' => $chatId,
                    'message_id' => $messageId,
                    'error' => $result['error']
                ]);
            }
            
            return $result;
            
        } catch (Exception $e) {
            $this->logger->error("Exception in deleteMessage", [
                'chat_id' => $chatId,
                'message_id' => $messageId,
                'error' => $e->getMessage()
            ]);
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Answer callback query
     */
    public function answerCallbackQuery($callbackQueryId, $text = '', $showAlert = false) {
        try {
            $url = "https://api.telegram.org/bot{$this->botToken}/answerCallbackQuery";
            $data = [
                'callback_query_id' => $callbackQueryId,
                'text' => $text,
                'show_alert' => $showAlert
            ];
            
            $result = $this->makeRequest($url, $data);
            
            if ($result['success']) {
                $this->logger->info("Callback query answered", [
                    'callback_query_id' => $callbackQueryId
                ]);
            }
            
            return $result;
            
        } catch (Exception $e) {
            $this->logger->error("Exception in answerCallbackQuery", [
                'callback_query_id' => $callbackQueryId,
                'error' => $e->getMessage()
            ]);
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Make HTTP request to Telegram API
     */
    private function makeRequest($url, $data, $timeout = 30) {
        try {
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($data));
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_TIMEOUT, $timeout);
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
            curl_setopt($ch, CURLOPT_USERAGENT, 'TelegramBot/1.0');
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $error = curl_error($ch);
            curl_close($ch);
            
            if ($error) {
                return [
                    'success' => false,
                    'error' => "cURL Error: {$error}",
                    'http_code' => $httpCode
                ];
            }
            
            if ($httpCode !== 200) {
                return [
                    'success' => false,
                    'error' => "HTTP Error: {$httpCode}",
                    'http_code' => $httpCode,
                    'response' => $response
                ];
            }
            
            $result = json_decode($response, true);
            
            if (!$result) {
                return [
                    'success' => false,
                    'error' => 'Invalid JSON response',
                    'response' => $response
                ];
            }
            
            if (!$result['ok']) {
                return [
                    'success' => false,
                    'error' => $result['description'] ?? 'Unknown API error',
                    'response' => $result
                ];
            }
            
            return [
                'success' => true,
                'result' => $result['result'],
                'response' => $result
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => "Request exception: {$e->getMessage()}"
            ];
        }
    }
    
    /**
     * Check if we should rate limit
     */
    private function shouldRateLimit($chatId) {
        $key = "rate_limit_{$chatId}";
        $now = time();
        
        if (!isset($this->rateLimiter[$key])) {
            $this->rateLimiter[$key] = ['count' => 0, 'window' => $now];
        }
        
        $rateData = $this->rateLimiter[$key];
        
        // Reset window if more than 1 minute
        if ($now - $rateData['window'] > 60) {
            $this->rateLimiter[$key] = ['count' => 1, 'window' => $now];
            return false;
        }
        
        // Check if we've exceeded rate limit (30 requests per minute)
        if ($rateData['count'] >= 30) {
            return true;
        }
        
        $this->rateLimiter[$key]['count']++;
        return false;
    }
}



