<?php
/**
 * User Management Class
 * Handles user operations with database
 */

class UserManager {
    private $db;
    private $logger;
    private $security;
    
    public function __construct($database, $logger, $security) {
        $this->db = $database;
        $this->logger = $logger;
        $this->security = $security;
    }
    
    /**
     * Save or update user information
     */
    public function saveUser($userId, $username, $firstName, $lastName) {
        try {
            $pdo = $this->db->getConnection();
            
            // Sanitize inputs
            $userId = $this->security->sanitizeInput($userId, 'int');
            $username = $this->security->sanitizeInput($username);
            $firstName = $this->security->sanitizeInput($firstName);
            $lastName = $this->security->sanitizeInput($lastName);
            
            $stmt = $pdo->prepare("
                INSERT INTO users (id, username, first_name, last_name, last_seen) 
                VALUES (?, ?, ?, ?, NOW()) 
                ON DUPLICATE KEY UPDATE 
                    username = VALUES(username),
                    first_name = VALUES(first_name),
                    last_name = VALUES(last_name),
                    last_seen = NOW()
            ");
            
            $result = $stmt->execute([$userId, $username, $firstName, $lastName]);
            
            if ($result) {
                $this->logger->info("User saved/updated", [
                    'user_id' => $userId,
                    'username' => $username
                ]);
            }
            
            return $result;
            
        } catch (Exception $e) {
            $this->logger->error("Failed to save user", [
                'user_id' => $userId,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }
    
    /**
     * Get all users
     */
    public function getAllUsers() {
        try {
            $pdo = $this->db->getConnection();
            $stmt = $pdo->prepare("SELECT * FROM users ORDER BY last_seen DESC");
            $stmt->execute();
            return $stmt->fetchAll();
        } catch (Exception $e) {
            $this->logger->error("Failed to get all users", ['error' => $e->getMessage()]);
            return [];
        }
    }
    
    /**
     * Get user by ID
     */
    public function getUserById($userId) {
        try {
            $pdo = $this->db->getConnection();
            $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
            $stmt->execute([$userId]);
            return $stmt->fetch();
        } catch (Exception $e) {
            $this->logger->error("Failed to get user by ID", [
                'user_id' => $userId,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }
    
    /**
     * Get user count
     */
    public function getUserCount() {
        try {
            $pdo = $this->db->getConnection();
            $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM users");
            $stmt->execute();
            $result = $stmt->fetch();
            return $result['count'];
        } catch (Exception $e) {
            $this->logger->error("Failed to get user count", ['error' => $e->getMessage()]);
            return 0;
        }
    }
    
    /**
     * Get active users (last seen within 7 days)
     */
    public function getActiveUsers($days = 7) {
        try {
            $pdo = $this->db->getConnection();
            $stmt = $pdo->prepare("
                SELECT * FROM users 
                WHERE last_seen > DATE_SUB(NOW(), INTERVAL ? DAY) 
                ORDER BY last_seen DESC
            ");
            $stmt->execute([$days]);
            return $stmt->fetchAll();
        } catch (Exception $e) {
            $this->logger->error("Failed to get active users", [
                'days' => $days,
                'error' => $e->getMessage()
            ]);
            return [];
        }
    }
    
    /**
     * Delete user
     */
    public function deleteUser($userId) {
        try {
            $pdo = $this->db->getConnection();
            $stmt = $pdo->prepare("DELETE FROM users WHERE id = ?");
            $result = $stmt->execute([$userId]);
            
            if ($result) {
                $this->logger->info("User deleted", ['user_id' => $userId]);
            }
            
            return $result;
        } catch (Exception $e) {
            $this->logger->error("Failed to delete user", [
                'user_id' => $userId,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }
    
    /**
     * Log user search
     */
    public function logUserSearch($userId, $username, $query, $resultsCount, $searchType = 'movie') {
        try {
            $pdo = $this->db->getConnection();
            
            // Sanitize inputs
            $userId = $this->security->sanitizeInput($userId, 'int');
            $username = $this->security->sanitizeInput($username);
            $query = $this->security->sanitizeInput($query);
            $resultsCount = $this->security->sanitizeInput($resultsCount, 'int');
            
            $stmt = $pdo->prepare("
                INSERT INTO user_searches (user_id, username, query, results_count, search_type) 
                VALUES (?, ?, ?, ?, ?)
            ");
            
            $result = $stmt->execute([$userId, $username, $query, $resultsCount, $searchType]);
            
            if ($result) {
                $this->logger->info("User search logged", [
                    'user_id' => $userId,
                    'query' => $query,
                    'results_count' => $resultsCount,
                    'search_type' => $searchType
                ]);
            }
            
            return $result;
        } catch (Exception $e) {
            $this->logger->error("Failed to log user search", [
                'user_id' => $userId,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }
    
    /**
     * Get search statistics
     */
    public function getSearchStats($days = 30) {
        try {
            $pdo = $this->db->getConnection();
            $stmt = $pdo->prepare("
                SELECT 
                    search_type,
                    COUNT(*) as total_searches,
                    COUNT(DISTINCT user_id) as unique_users,
                    AVG(results_count) as avg_results
                FROM user_searches 
                WHERE created_at > DATE_SUB(NOW(), INTERVAL ? DAY)
                GROUP BY search_type
            ");
            $stmt->execute([$days]);
            return $stmt->fetchAll();
        } catch (Exception $e) {
            $this->logger->error("Failed to get search stats", [
                'days' => $days,
                'error' => $e->getMessage()
            ]);
            return [];
        }
    }
}



