<?php
/**
 * Database Configuration
 * Handles database connection and basic operations
 */

class Database {
    private $pdo;
    private $host;
    private $dbname;
    private $username;
    private $password;
    
    public function __construct() {
        // Load hosting configuration first
        if (file_exists(__DIR__ . '/hosting_config.php')) {
            require_once __DIR__ . '/hosting_config.php';
        }
        
        // Load environment variables if not already loaded
        $this->loadEnvironmentVariables();
        
        // Try multiple methods to get database configuration
        $this->host = $this->getConfigValue('DB_HOST', 'localhost');
        $this->dbname = $this->getConfigValue('DB_NAME', 'telegram_bot');
        $this->username = $this->getConfigValue('DB_USER', 'root');
        $this->password = $this->getConfigValue('DB_PASS', '');
        
        // Debug: Log the database credentials being used
        error_log("Database connection attempt: Host={$this->host}, DB={$this->dbname}, User={$this->username}");
        
        $this->connect();
        $this->createTables();
    }
    
    private function getConfigValue($key, $default) {
        // Method 1: Check $_ENV
        if (isset($_ENV[$key])) {
            return $_ENV[$key];
        }
        
        // Method 2: Check server environment
        $value = getenv($key);
        if ($value !== false) {
            return $value;
        }
        
        // Method 3: Check $_SERVER
        if (isset($_SERVER[$key])) {
            return $_SERVER[$key];
        }
        
        // Method 4: Hardcoded values for cPanel hosting
        // Uncomment and modify these lines for your specific hosting
        /*
        switch ($key) {
            case 'DB_HOST':
                return 'localhost';
            case 'DB_NAME':
                return 'andn3765_bottele';
            case 'DB_USER':
                return 'andn3765_bottele';
            case 'DB_PASS':
                return 'andn3765_bottele';
        }
        */
        
        return $default;
    }
    
    private function loadEnvironmentVariables() {
        // Check if environment variables are already loaded
        if (isset($_ENV['DB_HOST'])) {
            return;
        }
        
        // Try multiple paths for .env file
        $envPaths = [
            '.env',
            '../.env',
            dirname(__DIR__) . '/.env',
            __DIR__ . '/../.env'
        ];
        
        foreach ($envPaths as $envPath) {
            if (file_exists($envPath)) {
                $lines = file($envPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
                foreach ($lines as $line) {
                    if (strpos($line, '=') !== false && strpos($line, '#') !== 0) {
                        list($key, $value) = explode('=', $line, 2);
                        $_ENV[trim($key)] = trim($value);
                    }
                }
                error_log("Environment variables loaded from: " . $envPath);
                break;
            }
        }
        
        // If still no environment variables, try to load from server environment
        if (!isset($_ENV['DB_HOST'])) {
            $_ENV['DB_HOST'] = getenv('DB_HOST') ?: 'localhost';
            $_ENV['DB_NAME'] = getenv('DB_NAME') ?: 'telegram_bot';
            $_ENV['DB_USER'] = getenv('DB_USER') ?: 'root';
            $_ENV['DB_PASS'] = getenv('DB_PASS') ?: '';
            error_log("Environment variables loaded from server environment");
        }
    }
    
    private function connect() {
        try {
            $dsn = "mysql:host={$this->host};dbname={$this->dbname};charset=utf8mb4";
            $this->pdo = new PDO($dsn, $this->username, $this->password, [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
            ]);
        } catch (PDOException $e) {
            throw new DatabaseException("Database connection failed: " . $e->getMessage());
        }
    }
    
    private function createTables() {
        $tables = [
            'users' => "
                CREATE TABLE IF NOT EXISTS users (
                    id BIGINT PRIMARY KEY,
                    username VARCHAR(255),
                    first_name VARCHAR(255),
                    last_name VARCHAR(255),
                    last_seen TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
            ",
            'admin_users' => "
                CREATE TABLE IF NOT EXISTS admin_users (
                    id BIGINT PRIMARY KEY,
                    username VARCHAR(255),
                    added_by BIGINT,
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
            ",
            'user_searches' => "
                CREATE TABLE IF NOT EXISTS user_searches (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    user_id BIGINT,
                    username VARCHAR(255),
                    query VARCHAR(500),
                    results_count INT,
                    search_type ENUM('movie', 'adult') DEFAULT 'movie',
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    INDEX idx_user_id (user_id),
                    INDEX idx_created_at (created_at)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
            ",
            'broadcast_logs' => "
                CREATE TABLE IF NOT EXISTS broadcast_logs (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    admin_id BIGINT,
                    message TEXT,
                    success_count INT,
                    failed_count INT,
                    total_users INT,
                    duration INT,
                    status ENUM('pending', 'running', 'completed', 'cancelled') DEFAULT 'pending',
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    completed_at TIMESTAMP NULL,
                    INDEX idx_admin_id (admin_id),
                    INDEX idx_status (status)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
            ",
            'rate_limits' => "
                CREATE TABLE IF NOT EXISTS rate_limits (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    user_id BIGINT,
                    action VARCHAR(50),
                    count INT DEFAULT 1,
                    window_start TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    expires_at TIMESTAMP,
                    INDEX idx_user_action (user_id, action),
                    INDEX idx_expires_at (expires_at)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
            ",
            'system_logs' => "
                CREATE TABLE IF NOT EXISTS system_logs (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    level ENUM('DEBUG', 'INFO', 'WARNING', 'ERROR', 'CRITICAL') DEFAULT 'INFO',
                    component VARCHAR(50) NOT NULL,
                    message TEXT NOT NULL,
                    user_id BIGINT NULL,
                    data JSON NULL,
                    ip_address VARCHAR(45) NULL,
                    user_agent TEXT NULL,
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    INDEX idx_level (level),
                    INDEX idx_component (component),
                    INDEX idx_user_id (user_id),
                    INDEX idx_created_at (created_at)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
            ",
            'bot_logs' => "
                CREATE TABLE IF NOT EXISTS bot_logs (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    log_type ENUM('access', 'debug', 'error', 'redirect', 'ad_click') NOT NULL,
                    user_id BIGINT NULL,
                    username VARCHAR(255) NULL,
                    action VARCHAR(100) NULL,
                    details TEXT NULL,
                    ip_address VARCHAR(45) NULL,
                    user_agent TEXT NULL,
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    INDEX idx_log_type (log_type),
                    INDEX idx_user_id (user_id),
                    INDEX idx_created_at (created_at)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
            ",
            'log_settings' => "
                CREATE TABLE IF NOT EXISTS log_settings (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    setting_name VARCHAR(100) UNIQUE NOT NULL,
                    setting_value TEXT NOT NULL,
                    description TEXT NULL,
                    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
            "
        ];
        
        foreach ($tables as $table => $sql) {
            try {
                $this->pdo->exec($sql);
            } catch (PDOException $e) {
                throw new DatabaseException("Failed to create table {$table}: " . $e->getMessage());
            }
        }
        
        // Insert default log settings
        $this->insertDefaultLogSettings();
    }
    
    private function insertDefaultLogSettings() {
        $defaultSettings = [
            ['auto_reset_hours', '24', 'Auto reset bot logs every N hours (0 = disabled)'],
            ['max_log_entries', '10000', 'Maximum log entries to keep'],
            ['log_retention_days', '30', 'Log retention period in days'],
            ['enable_debug_logs', '1', 'Enable debug logging (1 = yes, 0 = no)'],
            ['log_level', 'INFO', 'Minimum log level to store (DEBUG, INFO, WARNING, ERROR, CRITICAL)']
        ];
        
        foreach ($defaultSettings as $setting) {
            $stmt = $this->pdo->prepare("
                INSERT IGNORE INTO log_settings (setting_name, setting_value, description) 
                VALUES (?, ?, ?)
            ");
            $stmt->execute($setting);
        }
    }
    
    public function getConnection() {
        return $this->pdo;
    }
    
    public function beginTransaction() {
        return $this->pdo->beginTransaction();
    }
    
    public function commit() {
        return $this->pdo->commit();
    }
    
    public function rollback() {
        return $this->pdo->rollback();
    }
}

class DatabaseException extends Exception {}


