<?php
/**
 * Database Configuration - Alternative Method
 * This file provides multiple ways to load database configuration
 */

class DatabaseConfig {
    private static $config = null;
    
    public static function getConfig() {
        if (self::$config === null) {
            self::$config = self::loadConfig();
        }
        return self::$config;
    }
    
    private static function loadConfig() {
        $config = [
            'host' => 'localhost',
            'dbname' => 'telegram_bot',
            'username' => 'root',
            'password' => ''
        ];
        
        // Method 1: Try to load from .env file
        $envPaths = [
            '.env',
            '../.env',
            dirname(__DIR__) . '/.env',
            __DIR__ . '/../.env'
        ];
        
        foreach ($envPaths as $envPath) {
            if (file_exists($envPath)) {
                $lines = file($envPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
                foreach ($lines as $line) {
                    if (strpos($line, '=') !== false && strpos($line, '#') !== 0) {
                        list($key, $value) = explode('=', $line, 2);
                        $key = trim($key);
                        $value = trim($value);
                        
                        switch ($key) {
                            case 'DB_HOST':
                                $config['host'] = $value;
                                break;
                            case 'DB_NAME':
                                $config['dbname'] = $value;
                                break;
                            case 'DB_USER':
                                $config['username'] = $value;
                                break;
                            case 'DB_PASS':
                                $config['password'] = $value;
                                break;
                        }
                    }
                }
                error_log("Database config loaded from: " . $envPath);
                break;
            }
        }
        
        // Method 2: Try server environment variables
        if ($config['host'] === 'localhost') {
            $config['host'] = getenv('DB_HOST') ?: $config['host'];
            $config['dbname'] = getenv('DB_NAME') ?: $config['dbname'];
            $config['username'] = getenv('DB_USER') ?: $config['username'];
            $config['password'] = getenv('DB_PASS') ?: $config['password'];
        }
        
        // Method 3: Try $_ENV superglobal
        if (isset($_ENV['DB_HOST'])) {
            $config['host'] = $_ENV['DB_HOST'];
            $config['dbname'] = $_ENV['DB_NAME'];
            $config['username'] = $_ENV['DB_USER'];
            $config['password'] = $_ENV['DB_PASS'];
        }
        
        // Method 4: Hardcoded fallback for cPanel hosting
        // Uncomment and modify these lines for your cPanel hosting
        /*
        $config['host'] = 'localhost';
        $config['dbname'] = 'andn3765_bottele';
        $config['username'] = 'andn3765_bottele';
        $config['password'] = 'andn3765_bottele';
        */
        
        return $config;
    }
    
    public static function getHost() {
        return self::getConfig()['host'];
    }
    
    public static function getDbName() {
        return self::getConfig()['dbname'];
    }
    
    public static function getUsername() {
        return self::getConfig()['username'];
    }
    
    public static function getPassword() {
        return self::getConfig()['password'];
    }
    
    public static function getDsn() {
        $config = self::getConfig();
        return "mysql:host={$config['host']};dbname={$config['dbname']};charset=utf8mb4";
    }
    
    public static function debug() {
        $config = self::getConfig();
        return [
            'host' => $config['host'],
            'dbname' => $config['dbname'],
            'username' => $config['username'],
            'password' => empty($config['password']) ? '(empty)' : '(set)',
            'dsn' => self::getDsn()
        ];
    }
}
?>

