<?php
/**
 * Security and Rate Limiting System
 * Handles input sanitization, rate limiting, and security measures
 */

class Security {
    private $db;
    private $logger;
    
    public function __construct($database, $logger) {
        $this->db = $database;
        $this->logger = $logger;
    }
    
    /**
     * Sanitize input to prevent XSS and injection attacks
     */
    public function sanitizeInput($input, $type = 'string') {
        if (is_array($input)) {
            return array_map([$this, 'sanitizeInput'], $input);
        }
        
        $input = trim($input);
        
        switch ($type) {
            case 'string':
                return htmlspecialchars($input, ENT_QUOTES, 'UTF-8');
            case 'int':
                return (int) filter_var($input, FILTER_SANITIZE_NUMBER_INT);
            case 'float':
                return (float) filter_var($input, FILTER_SANITIZE_NUMBER_FLOAT, FILTER_FLAG_ALLOW_FRACTION);
            case 'email':
                return filter_var($input, FILTER_SANITIZE_EMAIL);
            case 'url':
                return filter_var($input, FILTER_SANITIZE_URL);
            case 'html':
                // Allow some HTML tags but sanitize dangerous ones
                $allowedTags = '<b><i><u><code><pre><a>';
                return strip_tags($input, $allowedTags);
            default:
                return htmlspecialchars($input, ENT_QUOTES, 'UTF-8');
        }
    }
    
    /**
     * Validate search query
     */
    public function validateSearchQuery($query) {
        $query = $this->sanitizeInput($query);
        
        // Check length
        if (strlen($query) < 2) {
            return [
                'valid' => false,
                'message' => '❌ Query terlalu pendek. Minimal 2 karakter.',
                'query' => $query
            ];
        }
        
        if (strlen($query) > 100) {
            return [
                'valid' => false,
                'message' => '❌ Query terlalu panjang. Maksimal 100 karakter.',
                'query' => $query
            ];
        }
        
        // Check for dangerous patterns
        $dangerousPatterns = [
            '/<script/i',
            '/javascript:/i',
            '/on\w+\s*=/i',
            '/eval\s*\(/i',
            '/expression\s*\(/i'
        ];
        
        foreach ($dangerousPatterns as $pattern) {
            if (preg_match($pattern, $query)) {
                $this->logger->warning("Dangerous pattern detected in query", ['query' => $query]);
                return [
                    'valid' => false,
                    'message' => '❌ Query mengandung karakter yang tidak diizinkan.',
                    'query' => $query
                ];
            }
        }
        
        // Check for valid characters (letters, numbers, spaces, common punctuation)
        if (!preg_match('/^[a-zA-Z0-9\s\-\:\'\.\,\!\(\)\?]+$/', $query)) {
            return [
                'valid' => false,
                'message' => '❌ Query hanya boleh mengandung huruf, angka, dan karakter yang diizinkan.',
                'query' => $query
            ];
        }
        
        return [
            'valid' => true,
            'message' => '',
            'query' => $query
        ];
    }
    
    /**
     * Rate limiting system
     */
    public function checkRateLimit($userId, $action, $maxAttempts = 10, $windowMinutes = 1) {
        try {
            $pdo = $this->db->getConnection();
            
            // Clean expired rate limits
            $stmt = $pdo->prepare("DELETE FROM rate_limits WHERE expires_at < NOW()");
            $stmt->execute();
            
            // Check current rate limit
            $stmt = $pdo->prepare("
                SELECT COUNT(*) as count 
                FROM rate_limits 
                WHERE user_id = ? AND action = ? AND window_start > DATE_SUB(NOW(), INTERVAL ? MINUTE)
            ");
            $stmt->execute([$userId, $action, $windowMinutes]);
            $result = $stmt->fetch();
            
            if ($result['count'] >= $maxAttempts) {
                $this->logger->warning("Rate limit exceeded", [
                    'user_id' => $userId,
                    'action' => $action,
                    'attempts' => $result['count']
                ]);
                return false;
            }
            
            // Record this attempt
            $stmt = $pdo->prepare("
                INSERT INTO rate_limits (user_id, action, window_start, expires_at) 
                VALUES (?, ?, NOW(), DATE_ADD(NOW(), INTERVAL ? MINUTE))
            ");
            $stmt->execute([$userId, $action, $windowMinutes]);
            
            return true;
            
        } catch (Exception $e) {
            $this->logger->error("Rate limit check failed", [
                'user_id' => $userId,
                'action' => $action,
                'error' => $e->getMessage()
            ]);
            return true; // Allow on error to prevent blocking users
        }
    }
    
    /**
     * Validate user input for broadcast messages
     */
    public function validateBroadcastMessage($message) {
        $message = $this->sanitizeInput($message, 'html');
        
        if (empty(trim($message))) {
            return [
                'valid' => false,
                'message' => '❌ Pesan broadcast tidak boleh kosong.',
                'cleaned_message' => $message
            ];
        }
        
        if (strlen($message) < 5) {
            return [
                'valid' => false,
                'message' => '❌ Pesan terlalu pendek. Minimal 5 karakter.',
                'cleaned_message' => $message
            ];
        }
        
        if (strlen($message) > 4000) {
            return [
                'valid' => false,
                'message' => '❌ Pesan terlalu panjang. Maksimal 4000 karakter.',
                'cleaned_message' => $message
            ];
        }
        
        // Check for spam patterns
        $spamPatterns = [
            '/(.)\1{10,}/', // Repeated characters
            '/https?:\/\/[^\s]+/', // URLs (might be spam)
            '/\b(click|here|now|free|win|prize|money)\b/i' // Spam keywords
        ];
        
        foreach ($spamPatterns as $pattern) {
            if (preg_match($pattern, $message)) {
                $this->logger->warning("Potential spam detected in broadcast", ['message' => $message]);
                return [
                    'valid' => false,
                    'message' => '❌ Pesan mengandung konten yang mencurigakan.',
                    'cleaned_message' => $message
                ];
            }
        }
        
        return [
            'valid' => true,
            'message' => '',
            'cleaned_message' => $message
        ];
    }
    
    /**
     * Check if user is admin
     */
    public function isAdmin($userId) {
        try {
            $pdo = $this->db->getConnection();
            $stmt = $pdo->prepare("SELECT id FROM admin_users WHERE id = ?");
            $stmt->execute([$userId]);
            return $stmt->fetch() !== false;
        } catch (Exception $e) {
            $this->logger->error("Admin check failed", [
                'user_id' => $userId,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }
    
    /**
     * Add admin user
     */
    public function addAdmin($userId, $username, $addedBy) {
        try {
            $pdo = $this->db->getConnection();
            $stmt = $pdo->prepare("
                INSERT INTO admin_users (id, username, added_by) 
                VALUES (?, ?, ?) 
                ON DUPLICATE KEY UPDATE username = VALUES(username)
            ");
            $stmt->execute([$userId, $username, $addedBy]);
            
            $this->logger->info("Admin user added", [
                'user_id' => $userId,
                'username' => $username,
                'added_by' => $addedBy
            ]);
            
            return true;
        } catch (Exception $e) {
            $this->logger->error("Failed to add admin", [
                'user_id' => $userId,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }
    
    /**
     * Remove admin user
     */
    public function removeAdmin($userId) {
        try {
            $pdo = $this->db->getConnection();
            $stmt = $pdo->prepare("DELETE FROM admin_users WHERE id = ?");
            $stmt->execute([$userId]);
            
            $this->logger->info("Admin user removed", ['user_id' => $userId]);
            return true;
        } catch (Exception $e) {
            $this->logger->error("Failed to remove admin", [
                'user_id' => $userId,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }
}



