<?php
/**
 * Download Links Extractor
 * File untuk menampilkan link download berdasarkan parameter URL film
 */

// Enable error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Set timezone ke Jakarta
date_default_timezone_set('Asia/Jakarta');

// Load environment variables
if (file_exists('../.env')) {
    $lines = file('../.env', FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    foreach ($lines as $line) {
        if (strpos($line, '=') !== false && strpos($line, '#') !== 0) {
            list($key, $value) = explode('=', $line, 2);
            $_ENV[trim($key)] = trim($value);
        }
    }
}

// Get domain configuration
$domain = $_ENV['DOMAIN'] ?? 'https://tes.apivalidasi.my.id';

// Initialize database and logger
require_once '../config/database.php';
require_once '../classes/DatabaseLogger.php';

try {
    $database = new Database();
    $logger = new DatabaseLogger($database->getConnection(), 'dl.php');
} catch (Exception $e) {
    // Fallback to file logging if database fails
    $logger = null;
}

// Fungsi helper untuk mendapatkan waktu Jakarta
function getJakartaTime($format = 'Y-m-d H:i:s') {
    return date($format);
}

// Log semua akses ke dl.php
$accessLog = getJakartaTime() . " - dl.php accessed with params: " . json_encode($_GET) . "\n";
if ($logger) {
    $logger->logBot('access', 'dl.php accessed', null, null, 'access', json_encode($_GET));
} else {
    file_put_contents('bot_log.txt', $accessLog, FILE_APPEND | LOCK_EX);
}

// Cek parameter URL
if (!isset($_GET['url']) || empty($_GET['url'])) {
    http_response_code(400);
    echo json_encode([
        'error' => 'Parameter URL diperlukan',
        'usage' => 'dl.php?url=film-slug&ref=encoded-ads-param'
    ]);
    exit;
}

$encodedSlug = $_GET['url'];
$encodedRef = isset($_GET['ref']) ? $_GET['ref'] : null;

// Log parameter yang diterima
$paramLog = getJakartaTime() . " - Parameters: url={$encodedSlug}, ref={$encodedRef}\n";
if ($logger) {
    $logger->logBot('debug', 'Received parameters', null, null, 'params', "url={$encodedSlug}, ref={$encodedRef}");
} else {
    file_put_contents('bot_log.txt', $paramLog, FILE_APPEND | LOCK_EX);
}

// Log untuk debug
file_put_contents('bot_log.txt', getJakartaTime() . " - dl.php processing request\n", FILE_APPEND | LOCK_EX);

// Decode slug dari base64
$movieSlug = base64_decode($encodedSlug);

// Decode ref parameter
$adsParam = null;
if ($encodedRef) {
    $decodedRef = base64_decode($encodedRef);
    $decodeLog = getJakartaTime() . " - Decoded ref: {$decodedRef}\n";
    file_put_contents('bot_log.txt', $decodeLog, FILE_APPEND | LOCK_EX);
    
    if ($decodedRef === 'ads1') {
        $adsParam = '1';
    } elseif ($decodedRef === 'ads2') {
        $adsParam = '2';
    } elseif ($decodedRef === 'ads3') {
        $adsParam = '3';
    } elseif ($decodedRef === 'adsinirandom') {
        // Skip logging untuk random parameter
        $adsParam = null;
    }
}

// Cek apakah ini adult content URL (embed URL) atau movie slug
$isAdultContent = false;
$movieUrl = '';

if (filter_var($movieSlug, FILTER_VALIDATE_URL)) {
    // Ini adalah adult content embed URL
    $isAdultContent = true;
    $movieUrl = $movieSlug;
    file_put_contents('bot_log.txt', getJakartaTime() . " - Adult content URL detected: {$movieUrl}\n", FILE_APPEND | LOCK_EX);
} else {
    // Ini adalah movie slug biasa
    if (empty($movieSlug) || !preg_match('/^[a-zA-Z0-9\-]+$/', $movieSlug)) {
        http_response_code(400);
        echo json_encode([
            'error' => 'Invalid encoded URL',
            'encoded' => $encodedSlug
        ]);
        exit;
    }
    
    // Buat URL lengkap dari slug
    $movieUrl = 'https://tv6.lk21official.cc/' . $movieSlug;
    file_put_contents('bot_log.txt', getJakartaTime() . " - Movie URL: {$movieUrl}\n", FILE_APPEND | LOCK_EX);
}

// Load ads configuration from JSON file
$adsConfigFile = '../ads_config.json';
$defaultAdsUrls = [
    '1' => 'https://example.com/ads1',
    '2' => 'https://example.com/ads2',
    '3' => 'https://example.com/ads3'
];

// Adult content specific ads (jika ada)
$adultAdsUrls = [
    '1' => 'https://example.com/adult-ads1',
    '2' => 'https://example.com/adult-ads2',
    '3' => 'https://example.com/adult-ads3'
];

// Load ads config
$adsUrls = $defaultAdsUrls;
if (file_exists($adsConfigFile)) {
    $config = json_decode(file_get_contents($adsConfigFile), true);
    if ($config) {
        $adsUrls = $config;
    }
}

// Log status ads parameter
$adsStatusLog = getJakartaTime() . " - Ads parameter: {$adsParam}, Movie: {$movieSlug}\n";
file_put_contents('bot_log.txt', $adsStatusLog, FILE_APPEND | LOCK_EX);

// Cek parameter ads dan redirect jika ada
if ($adsParam && isset($adsUrls[$adsParam])) {
    // Log klik ads - selalu log untuk tracking yang akurat
    $userId = 'web_user';
    $username = 'web_user';
    $adType = 'ads' . $adsParam;
    
    // Log ke database atau file
    $timestamp = getJakartaTime();
    if ($logger) {
        $logger->logBot('ad_click', 'Ad clicked', $userId, $username, 'ad_click', "ad_type={$adType}, movie={$movieSlug}");
    } else {
        // Fallback to file logging
        $logFile = 'admin_logs.txt';
        $logEntry = "{$timestamp}|AD_CLICK|{$userId}|{$username}|{$adType}|{$movieSlug}\n";
        file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
        
        $debugLog = "dl.php - AD CLICK: {$adType} for movie: {$movieSlug} at {$timestamp}\n";
        file_put_contents('bot_log.txt', $debugLog, FILE_APPEND | LOCK_EX);
    }
    
    // Log redirect URL
    $redirectLog = getJakartaTime() . " - Redirecting to ads URL: {$adsUrls[$adsParam]}\n";
    if ($logger) {
        $logger->logBot('redirect', 'Redirecting to ads', null, null, 'redirect', "url={$adsUrls[$adsParam]}");
    } else {
        file_put_contents('bot_log.txt', $redirectLog, FILE_APPEND | LOCK_EX);
    }
    
    header('Location: ' . $adsUrls[$adsParam]);
    exit;
} else {
    // Log jika tidak ada ads parameter atau URL tidak ditemukan
    $noAdsLog = getJakartaTime() . " - No ads redirect: adsParam={$adsParam}, available URLs=" . json_encode(array_keys($adsUrls)) . "\n";
    file_put_contents('bot_log.txt', $noAdsLog, FILE_APPEND | LOCK_EX);
}

// Scrape halaman film
function scrapeMoviePage($url) {
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    curl_setopt($ch, CURLOPT_ENCODING, '');
    
    $html = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode !== 200 || !$html) {
        return false;
    }
    
    return $html;
}

// Extract download links
function extractDownloadLinks($html) {
    $dom = new DOMDocument();
    libxml_use_internal_errors(true);
    @$dom->loadHTML($html);
    libxml_clear_errors();
    
    $xpath = new DOMXPath($dom);
    
    $downloadLinks = [];
    
    // Cari div player-options
    $playerOptions = $xpath->query('//div[@class="player-options"]');
    
    if ($playerOptions->length > 0) {
        // Cari semua link dalam player-list
        $playerLinks = $xpath->query('.//ul[@id="player-list"]//a');
        
        foreach ($playerLinks as $link) {
            $href = $link->getAttribute('href');
            $dataUrl = $link->getAttribute('data-url');
            $dataServer = $link->getAttribute('data-server');
            $text = trim($link->textContent);
            
            if (!empty($dataUrl) && !empty($dataServer)) {
                $downloadLinks[] = [
                    'server' => strtoupper($dataServer),
                    'name' => $text,
                    'url' => $dataUrl,
                    'iframe_url' => $href
                ];
            }
        }
        
        // Cari juga dari select option
        $selectOptions = $xpath->query('//select[@id="player-select"]//option');
        
        foreach ($selectOptions as $option) {
            $value = $option->getAttribute('value');
            $dataServer = $option->getAttribute('data-server');
            $text = trim($option->textContent);
            
            if (!empty($value) && !empty($dataServer)) {
                // Cek apakah sudah ada di array
                $exists = false;
                foreach ($downloadLinks as $existing) {
                    if ($existing['url'] === $value) {
                        $exists = true;
                        break;
                    }
                }
                
                if (!$exists) {
                    $downloadLinks[] = [
                        'server' => strtoupper($dataServer),
                        'name' => $text,
                        'url' => $value,
                        'iframe_url' => ''
                    ];
                }
            }
        }
    }
    
    // Urutkan berdasarkan prioritas: P2P → TURBOVIP → CAST → HYDRAX
    $priority = ['P2P', 'TURBOVIP', 'CAST', 'HYDRAX'];
    $sortedLinks = [];
    
    foreach ($priority as $server) {
        foreach ($downloadLinks as $link) {
            if ($link['server'] === $server) {
                $sortedLinks[] = $link;
                break;
            }
        }
    }
    
    // Tambahkan link lain yang tidak ada di priority
    foreach ($downloadLinks as $link) {
        if (!in_array($link['server'], $priority)) {
            $sortedLinks[] = $link;
        }
    }
    
    // Kembalikan hanya 1 link (yang pertama/terprioritas)
    return !empty($sortedLinks) ? [$sortedLinks[0]] : [];
}

// Handle adult content vs movie content differently
if ($isAdultContent) {
    // Untuk adult content, cek parameter ads dan redirect sesuai
    // Adult content menggunakan sistem 3 button random (1 embed + 2 ads)
    if ($logger) {
        $logger->logBot('debug', 'Adult content detected', null, null, 'adult_content', "url={$movieUrl}");
    } else {
        file_put_contents('bot_log.txt', getJakartaTime() . " - Adult content URL detected: {$movieUrl}\n", FILE_APPEND | LOCK_EX);
    }
    
    // Cek parameter ads dan redirect jika ada
    if ($adsParam && isset($adsUrls[$adsParam])) {
        // Log klik ads untuk adult content
        $userId = 'web_user';
        $username = 'web_user';
        $adType = 'adult_ads' . $adsParam;
        
        // Log ke database atau file
        $timestamp = getJakartaTime();
        if ($logger) {
            $logger->logBot('ad_click', 'Adult ad clicked', $userId, $username, 'adult_ad_click', "ad_type={$adType}");
        } else {
            // Fallback to file logging
            $logFile = 'admin_logs.txt';
            $logEntry = "{$timestamp}|AD_CLICK|{$userId}|{$username}|{$adType}|adult_content\n";
            file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
            
            $debugLog = "dl.php - ADULT AD CLICK: {$adType} for adult content at {$timestamp}\n";
            file_put_contents('bot_log.txt', $debugLog, FILE_APPEND | LOCK_EX);
        }
        
        // Log redirect URL
        $redirectLog = getJakartaTime() . " - Redirecting adult content to ads URL: {$adsUrls[$adsParam]}\n";
        file_put_contents('bot_log.txt', $redirectLog, FILE_APPEND | LOCK_EX);
        
        header('Location: ' . $adsUrls[$adsParam]);
        exit;
    } else {
        // Jika tidak ada ads parameter, redirect ke embed URL langsung
        // Ini adalah button yang valid (1 dari 3 button random)
        file_put_contents('bot_log.txt', getJakartaTime() . " - Redirecting to adult content embed: {$movieUrl}\n", FILE_APPEND | LOCK_EX);
        header('Location: ' . $movieUrl);
        exit;
    }
} else {
    // Untuk movie content, scrape halaman film
    $html = scrapeMoviePage($movieUrl);

    if ($html === false) {
        http_response_code(500);
        echo json_encode([
            'error' => 'Gagal mengambil halaman film',
            'url' => $movieUrl
        ]);
        exit;
    }

    // Extract download links
    $downloadLinks = extractDownloadLinks($html);

    if (empty($downloadLinks)) {
        echo json_encode([
            'error' => 'Tidak ada link download ditemukan',
            'url' => $movieUrl,
            'message' => 'Film mungkin belum tersedia atau struktur halaman berubah'
        ]);
        exit;
    }

    // Set header untuk JSON
    header('Content-Type: application/json');

    // Tampilkan hasil - redirect ke download link
    if (!empty($downloadLinks)) {
        $link = $downloadLinks[0];
        
        // Khusus untuk P2P, ubah URL ke proxy
        if ($link['server'] === 'P2P') {
            // Extract ID dari URL asli
            $originalUrl = $link['url'];
            if (strpos($originalUrl, 'https://cloud.hownetwork.xyz/video.php?id=') === 0) {
                $id = substr($originalUrl, strlen('https://cloud.hownetwork.xyz/video.php?id='));
                $downloadUrl = $domain . '/public/player.php?id=' . $id;
            } else {
                $downloadUrl = $originalUrl;
            }
        } else {
            $downloadUrl = $link['url'];
        }
        
        // Redirect ke download link
        header('Location: ' . $downloadUrl);
        exit;
    } else {
        // Jika tidak ada download link, tampilkan error
        http_response_code(404);
        echo json_encode([
            'error' => 'Tidak ada link download tersedia',
            'url' => $movieUrl,
            'slug' => $movieSlug
        ]);
    }
}
?>