<?php
/**
 * Async Broadcast Script
 * Handles broadcast processing in background
 */

// Load required files
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/logger.php';
require_once __DIR__ . '/../config/security.php';
require_once __DIR__ . '/../classes/UserManager.php';
require_once __DIR__ . '/../classes/TelegramAPI.php';
require_once __DIR__ . '/../classes/BroadcastManager.php';

// Get command line arguments
$broadcastId = $argv[1] ?? null;
$adminChatId = $argv[2] ?? null;

if (!$broadcastId || !$adminChatId) {
    exit(1);
}

try {
    // Initialize components
    $db = new Database();
    $logger = new Logger();
    $security = new Security($db, $logger);
    $telegram = new TelegramAPI($_ENV['BOT_TOKEN'] ?? '', $logger);
    $userManager = new UserManager($db, $logger, $security);
    $broadcastManager = new BroadcastManager($db, $logger, $security, $telegram);
    
    $logger->info("Starting async broadcast", [
        'broadcast_id' => $broadcastId,
        'admin_chat_id' => $adminChatId
    ]);
    
    // Get broadcast details
    $broadcast = $broadcastManager->getBroadcastStatus($broadcastId);
    if (!$broadcast) {
        $logger->error("Broadcast not found", ['broadcast_id' => $broadcastId]);
        exit(1);
    }
    
    // Get all users
    $users = $userManager->getAllUsers();
    $totalUsers = count($users);
    $successCount = 0;
    $failCount = 0;
    $startTime = time();
    
    $logger->info("Broadcast started", [
        'broadcast_id' => $broadcastId,
        'total_users' => $totalUsers
    ]);
    
    // Send initial progress message
    $progressMessageId = null;
    $progressText = "🔄 <b>Broadcasting...</b>\n\n";
    $progressText .= "📊 <b>Progress:</b> 0/0 batch\n";
    $progressText .= "👥 <b>User:</b> 0/{$totalUsers}\n";
    $progressText .= "✅ <b>Berhasil:</b> 0\n";
    $progressText .= "❌ <b>Gagal:</b> 0\n";
    $progressText .= "⏱️ <b>Waktu:</b> 0s\n\n";
    $progressText .= "📈 <b>Progress:</b> ░░░░░░░░░░░░░░░░░░░░ 0%";
    
    $cancelButton = [[
        [
            'text' => '❌ Batalkan Broadcast',
            'callback_data' => 'broadcast_cancel_live'
        ]
    ]];
    
    $result = $telegram->sendMessageWithButtons($adminChatId, $progressText, $cancelButton);
    if ($result['success']) {
        $progressMessageId = $result['result']['message_id'];
    }
    
    // Process users with rate limiting
    $messageCount = 0;
    $rateLimitCount = 0;
    
    foreach ($users as $userIndex => $user) {
        $chatId = $user['id'];
        $messageCount++;
        
        // Check if broadcast was cancelled
        $currentBroadcast = $broadcastManager->getBroadcastStatus($broadcastId);
        if ($currentBroadcast && $currentBroadcast['status'] === 'cancelled') {
            $logger->info("Broadcast cancelled", ['broadcast_id' => $broadcastId]);
            break;
        }
        
        // Update progress every 10 messages
        if ($messageCount % 10 === 0 || $messageCount === 1) {
            if ($progressMessageId) {
                $progressText = "🔄 <b>Broadcasting...</b>\n\n";
                $progressText .= "👥 <b>User:</b> {$messageCount}/{$totalUsers}\n";
                $progressText .= "✅ <b>Berhasil:</b> {$successCount}\n";
                $progressText .= "❌ <b>Gagal:</b> {$failCount}\n";
                $progressText .= "⏱️ <b>Waktu:</b> " . (time() - $startTime) . "s\n\n";
                
                // Progress bar
                $progressBar = "";
                $progressPercent = round(($messageCount / $totalUsers) * 100);
                $progressBars = round($progressPercent / 5);
                for ($i = 0; $i < 20; $i++) {
                    if ($i < $progressBars) {
                        $progressBar .= "█";
                    } else {
                        $progressBar .= "░";
                    }
                }
                $progressText .= "📈 <b>Progress:</b> {$progressBar} {$progressPercent}%";
                
                $telegram->editMessage($adminChatId, $progressMessageId, $progressText);
            }
        }
        
        // Check rate limit
        if (!$security->checkRateLimit($chatId, 'broadcast', 1, 1)) {
            $failCount++;
            continue;
        }
        
        $result = $telegram->sendMessage($chatId, $broadcast['message']);
        
        if ($result['success']) {
            $successCount++;
            $rateLimitCount++;
        } else {
            $failCount++;
            $logger->warning("Failed to send broadcast to user", [
                'user_id' => $chatId,
                'error' => $result['error'] ?? 'Unknown error'
            ]);
        }
        
        // Rate limiting: delay 2 seconds every 20 messages
        if ($rateLimitCount % 20 === 0 && $messageCount < $totalUsers) {
            $logger->info("Rate limit delay", [
                'messages_sent' => $rateLimitCount,
                'total_messages' => $messageCount
            ]);
            sleep(2);
        }
        
        // Small delay to avoid rate limiting
        usleep(100000); // 0.1 second
        
        // Update database progress every 50 messages
        if ($messageCount % 50 === 0) {
            $broadcastManager->updateBroadcastProgress($broadcastId, $successCount, $failCount);
        }
    }
    
    // Final database update
    $broadcastManager->updateBroadcastProgress($broadcastId, $successCount, $failCount);
    
    // Send final result
    $duration = time() - $startTime;
    $status = $currentBroadcast && $currentBroadcast['status'] === 'cancelled' ? 'cancelled' : 'completed';
    
    if ($status === 'cancelled') {
        $finalText = "❌ <b>Broadcast Dibatalkan!</b>\n\n";
        $finalText .= "📊 <b>Hasil Sebelum Dibatalkan:</b>\n";
        $finalText .= "✅ <b>Berhasil:</b> {$successCount}\n";
        $finalText .= "❌ <b>Gagal:</b> {$failCount}\n";
        $finalText .= "👥 <b>Total:</b> {$totalUsers}\n";
        $finalText .= "⏱️ <b>Waktu:</b> {$duration}s\n";
        $finalText .= "📈 <b>Progress:</b> " . round(($successCount / $totalUsers) * 100, 1) . "%";
    } else {
        $finalText = "✅ <b>Broadcast Selesai!</b>\n\n";
        $finalText .= "📊 <b>Hasil Akhir:</b>\n";
        $finalText .= "✅ <b>Berhasil:</b> {$successCount}\n";
        $finalText .= "❌ <b>Gagal:</b> {$failCount}\n";
        $finalText .= "👥 <b>Total:</b> {$totalUsers}\n";
        $finalText .= "⏱️ <b>Waktu:</b> {$duration}s\n";
        $finalText .= "📈 <b>Success Rate:</b> " . round(($successCount / $totalUsers) * 100, 1) . "%";
    }
    
    if ($progressMessageId) {
        $telegram->editMessage($adminChatId, $progressMessageId, $finalText);
    }
    
    // Update final status
    $broadcastManager->updateBroadcastProgress($broadcastId, $successCount, $failCount, $status);
    
    $logger->info("Broadcast completed", [
        'broadcast_id' => $broadcastId,
        'success_count' => $successCount,
        'fail_count' => $failCount,
        'duration' => $duration,
        'status' => $status
    ]);
    
} catch (Exception $e) {
    $logger->error("Async broadcast failed", [
        'broadcast_id' => $broadcastId,
        'error' => $e->getMessage(),
        'trace' => $e->getTraceAsString()
    ]);
    exit(1);
}


