<?php
/**
 * Auto Reset Logs Script
 * Runs as cron job to automatically clean up logs based on settings
 */

// Set timezone
date_default_timezone_set('Asia/Jakarta');

// Load dependencies
require_once '../config/database.php';
require_once '../classes/DatabaseLogger.php';

try {
    $database = new Database();
    $logger = new DatabaseLogger($database->getConnection(), 'auto_reset');
    
    // Get log settings
    $settings = $logger->getLogSettings();
    $autoResetHours = (int)($settings['auto_reset_hours'] ?? 0);
    
    if ($autoResetHours <= 0) {
        echo "Auto reset disabled (hours: {$autoResetHours})\n";
        exit(0);
    }
    
    // Check if it's time to reset
    $lastResetFile = '../logs/last_reset.txt';
    $shouldReset = false;
    
    if (!file_exists($lastResetFile)) {
        $shouldReset = true;
    } else {
        $lastReset = (int)file_get_contents($lastResetFile);
        $hoursSinceReset = (time() - $lastReset) / 3600;
        
        if ($hoursSinceReset >= $autoResetHours) {
            $shouldReset = true;
        }
    }
    
    if ($shouldReset) {
        echo "Starting auto log cleanup...\n";
        
        // Perform cleanup
        $result = $logger->autoCleanup();
        
        if ($result) {
            // Update last reset time
            if (!is_dir('../logs')) {
                mkdir('../logs', 0755, true);
            }
            file_put_contents($lastResetFile, time());
            
            echo "Auto cleanup completed successfully\n";
            
            // Log the cleanup action
            $logger->log('INFO', 'Auto log cleanup completed', [
                'auto_reset_hours' => $autoResetHours,
                'timestamp' => date('Y-m-d H:i:s')
            ]);
        } else {
            echo "Auto cleanup failed\n";
            $logger->log('ERROR', 'Auto log cleanup failed', [
                'auto_reset_hours' => $autoResetHours,
                'timestamp' => date('Y-m-d H:i:s')
            ]);
        }
    } else {
        $nextReset = $autoResetHours - $hoursSinceReset;
        echo "Auto reset not due yet. Next reset in " . round($nextReset, 1) . " hours\n";
    }
    
} catch (Exception $e) {
    echo "Error in auto reset script: " . $e->getMessage() . "\n";
    error_log("Auto reset script error: " . $e->getMessage());
    exit(1);
}

