<?php
/**
 * Telegram Bot - New Modular Version
 * Improved with security, performance, and proper error handling
 */

// Enable error reporting for development
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Set timezone
date_default_timezone_set('Asia/Jakarta');

// Load environment variables
if (file_exists('.env')) {
    $lines = file('.env', FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    foreach ($lines as $line) {
        if (strpos($line, '=') !== false && strpos($line, '#') !== 0) {
            list($key, $value) = explode('=', $line, 2);
            $_ENV[trim($key)] = trim($value);
        }
    }
}

// Load required files
require_once 'config/database.php';
require_once 'config/logger.php';
require_once 'config/security.php';
require_once 'classes/UserManager.php';
require_once 'classes/TelegramAPI.php';
require_once 'classes/DatabaseLogger.php';
require_once 'classes/BroadcastManager.php';
require_once 'classes/MovieScraper.php';
require_once 'classes/AdultScraper.php';
require_once 'admin/integration/telegram_integration.php';

// Load configuration
$config = require 'config.php';

// Get domain configuration from config
$publicDomain = $config['public_domain'] ?? 'https://tes.apivalidasi.my.id/public';

// Initialize components
try {
    $db = new Database();
    $logger = new Logger();
    $security = new Security($db, $logger);
    $telegram = new TelegramAPI($config['bot_token'], $logger);
    $userManager = new UserManager($db, $logger, $security);
    $broadcastManager = new BroadcastManager($db, $logger, $security, $telegram);
    $movieScraper = new MovieScraper($config, $logger);
    $adultScraper = new AdultScraper($logger);
    
    $logger->info("Bot components initialized successfully");
    
} catch (Exception $e) {
    error_log("Failed to initialize bot: " . $e->getMessage());
    http_response_code(500);
    exit('Bot initialization failed');
}

/**
 * Main Bot Handler Class
 */
class BotHandler {
    private $db;
    private $logger;
    private $dbLogger;
    private $security;
    private $telegram;
    private $userManager;
    private $broadcastManager;
    private $movieScraper;
    private $adultScraper;
    private $publicDomain;
    
    public function __construct($db, $logger, $security, $telegram, $userManager, $broadcastManager, $movieScraper, $adultScraper, $publicDomain) {
        $this->db = $db;
        $this->logger = $logger;
        $this->dbLogger = new DatabaseLogger($db, 'telegram_bot');
        $this->security = $security;
        $this->telegram = $telegram;
        $this->userManager = $userManager;
        $this->broadcastManager = $broadcastManager;
        $this->movieScraper = $movieScraper;
        $this->adultScraper = $adultScraper;
        $this->publicDomain = $publicDomain;
    }
    
    /**
     * Process incoming webhook
     */
    public function processWebhook() {
        try {
            $input = file_get_contents('php://input');
            
            if (empty($input)) {
                $this->logger->warning("Empty webhook input");
                return;
            }
            
            $update = json_decode($input, true);
            
            if (!$update) {
                $this->logger->error("Invalid JSON in webhook", ['input' => $input]);
                http_response_code(400);
                return;
            }
            
            $this->logger->info("Webhook received", ['update_type' => $this->getUpdateType($update)]);
            
            // Process message
            if (isset($update['message'])) {
                $this->handleMessage($update['message']);
            }
            
            // Process callback query
            if (isset($update['callback_query'])) {
                $this->handleCallbackQuery($update['callback_query']);
            }
            
        } catch (Exception $e) {
            // Log error to database
            $this->dbLogger->log('ERROR', 'Webhook processing failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            $this->logger->error("Webhook processing failed", [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            http_response_code(500);
        }
    }
    
    /**
     * Handle incoming messages
     */
    private function handleMessage($message) {
        try {
            $chatId = $message['chat']['id'];
            $text = $message['text'] ?? '';
            $userId = $message['from']['id'];
            $username = $message['from']['username'] ?? '';
            $firstName = $message['from']['first_name'] ?? '';
            $lastName = $message['from']['last_name'] ?? '';
            
            // Rate limiting
            if (!$this->security->checkRateLimit($userId, 'message', 30, 1)) {
                $this->logger->warning("Rate limit exceeded", ['user_id' => $userId]);
                return;
            }
            
            // Save user
            $this->userManager->saveUser($userId, $username, $firstName, $lastName);
            
            // Log to database
            $this->dbLogger->logBot('access', 'Message received', $userId, $username, 'message', "text: {$text}");
            
            $this->logger->info("Message received", [
                'user_id' => $userId,
                'chat_id' => $chatId,
                'text' => substr($text, 0, 100)
            ]);
            
            // Handle commands
            if (strpos($text, '/cari') === 0) {
                $this->handleSearchCommand($chatId, $userId, $text, $message);
            } elseif (strpos($text, '/adult') === 0) {
                $this->handleAdultCommand($chatId, $userId, $text, $message);
            } elseif (strpos($text, '/broadcast') === 0) {
                $this->handleBroadcastCommand($chatId, $userId, $text);
            } elseif ($text === '/broadcast_confirm') {
                $this->handleBroadcastConfirm($chatId, $userId);
            } elseif ($text === '/broadcast_cancel') {
                $this->handleBroadcastCancel($chatId, $userId);
            } elseif ($text === '/stats') {
                $this->handleStatsCommand($chatId, $userId);
            } elseif ($text === '/start' || $text === '/help') {
                $this->handleStartHelpCommand($chatId, $userId, $text, $firstName, $username);
            } elseif (!empty($text) && $text[0] !== '/') {
                $this->handleTextMessage($chatId);
            }
            
        } catch (Exception $e) {
            $this->logger->error("Message handling failed", [
                'chat_id' => $chatId,
                'error' => $e->getMessage()
            ]);
        }
    }
    
    /**
     * Handle search command
     */
    private function handleSearchCommand($chatId, $userId, $text, $message) {
        try {
            $rawQuery = trim(substr($text, 6));
            
            // Validate query
            $validation = $this->security->validateSearchQuery($rawQuery);
            if (!$validation['valid']) {
                $this->telegram->sendMessage($chatId, $validation['message'] . "\n\n💡 <b>Contoh:</b> <code>/cari avengers</code>");
                return;
            }
            
            $query = $validation['query'];
            
            // Rate limiting for search
            if (!$this->security->checkRateLimit($userId, 'search', 10, 1)) {
                $this->telegram->sendMessage($chatId, "⏰ <b>Terlalu banyak pencarian!</b>\n\n💡 <b>Tunggu sebentar sebelum mencari lagi</b>");
                return;
            }
            
            // Scrape movies
            $movies = $this->movieScraper->scrapeMovies($query);
            
            if ($movies === false) {
                $this->telegram->sendMessage($chatId, "❌ <b>Gagal mengambil data</b>\n\n💡 <b>Coba lagi:</b> <code>/cari [judul film]</code>");
                return;
            }
            
            // Log search
            $username = $message['from']['username'] ?? 'unknown';
            $this->userManager->logUserSearch($userId, $username, $query, count($movies), 'movie');
            
            // Log to database
            $this->dbLogger->logBot('debug', 'Movie search performed', $userId, $username, 'search', "query: {$query}, results: " . count($movies));
            
            // Format and send results
            $formattedData = $this->formatMovieMessageWithButtons($movies, $query);
            $this->telegram->sendMessageWithButtons($chatId, $formattedData['text'], $formattedData['buttons'], $message['message_id']);
            
        } catch (Exception $e) {
            $this->logger->error("Search command failed", [
                'chat_id' => $chatId,
                'query' => $rawQuery,
                'error' => $e->getMessage()
            ]);
            $this->telegram->sendMessage($chatId, "❌ <b>Terjadi kesalahan saat mencari</b>");
        }
    }
    
    /**
     * Handle adult command
     */
    private function handleAdultCommand($chatId, $userId, $text, $message) {
        try {
            $rawQuery = trim(substr($text, 7));
            
            // Rate limiting for adult search
            if (!$this->security->checkRateLimit($userId, 'adult_search', 5, 1)) {
                $this->telegram->sendMessage($chatId, "⏰ <b>Terlalu banyak pencarian adult!</b>\n\n💡 <b>Tunggu sebentar sebelum mencari lagi</b>");
                return;
            }
            
            if ($rawQuery === 'random') {
                $this->handleAdultRandom($chatId, $userId, $message);
            } else {
                $this->handleAdultSearch($chatId, $userId, $rawQuery, $message);
            }
            
        } catch (Exception $e) {
            $this->logger->error("Adult command failed", [
                'chat_id' => $chatId,
                'query' => $rawQuery,
                'error' => $e->getMessage()
            ]);
            $this->telegram->sendMessage($chatId, "❌ <b>Terjadi kesalahan saat mencari konten adult</b>");
        }
    }
    
    /**
     * Handle adult random
     */
    private function handleAdultRandom($chatId, $userId, $message) {
        $this->telegram->sendMessage($chatId, "🔍 <b>Mengambil 10 adult content random...</b>");
        
        $randomAdultContent = $this->adultScraper->getRandomAdultContent();
        
        if (empty($randomAdultContent)) {
            $this->telegram->sendMessage($chatId, "❌ <b>Gagal mengambil adult content random</b>\n\n💡 <b>Coba lagi:</b> <code>/adult random</code>");
            return;
        }
        
        // Log search
        $username = $message['from']['username'] ?? 'unknown';
        $this->userManager->logUserSearch($userId, $username, "adult_random", count($randomAdultContent), 'adult');
        
        // Log to database
        $this->dbLogger->logBot('debug', 'Adult random search performed', $userId, $username, 'adult_random', "results: " . count($randomAdultContent));
        
        // Format message
        $messageText = "🔞 <b>Adult Random (10 Data)</b>\n\n";
        $messageText .= "📊 <b>Total:</b> " . count($randomAdultContent) . " data\n\n";
        
        $buttons = [];
        foreach ($randomAdultContent as $index => $content) {
            $buttonNumber = $index + 1;
            $messageText .= "{$buttonNumber}. <b>" . htmlspecialchars($content['title']) . "</b>\n";
            $messageText .= "   📅 {$content['Tahun']} | 🎬 {$content['Kualitas']} | ⏱️ {$content['Durasi']}\n\n";
            
            $buttons[] = [
                'text' => $buttonNumber,
                'callback_data' => 'adult_' . $content['id']
            ];
        }
        
        $buttons[] = [
            'text' => '🎲 Random Lagi',
            'callback_data' => 'adult_random_new'
        ];
        
        $buttonGrid = $this->createButtonGrid($buttons, 5);
        $this->telegram->sendMessageWithButtons($chatId, $messageText, $buttonGrid);
    }
    
    /**
     * Handle adult search
     */
    private function handleAdultSearch($chatId, $userId, $query, $message) {
        // Validate query
        $validation = $this->security->validateSearchQuery($query);
        if (!$validation['valid']) {
            $this->telegram->sendMessage($chatId, $validation['message'] . "\n\n💡 <b>Contoh:</b> <code>/adult JAV</code>");
            return;
        }
        
        $query = $validation['query'];
        
        // Scrape adult content
        $adultContent = $this->adultScraper->scrapeAdultContent($query);
        
        if ($adultContent === false) {
            $this->telegram->sendMessage($chatId, "❌ <b>Gagal mengambil data adult</b>\n\n💡 <b>Coba lagi:</b> <code>/adult [query]</code>");
            return;
        }
        
        // Log search
        $username = $message['from']['username'] ?? 'unknown';
        $this->userManager->logUserSearch($userId, $username, "adult_" . $query, count($adultContent), 'adult');
        
        // Log to database
        $this->dbLogger->logBot('debug', 'Adult search performed', $userId, $username, 'adult_search', "query: {$query}, results: " . count($adultContent));
        
        // Format and send results
        $formattedData = $this->formatAdultContentMessageWithButtons($adultContent, $query);
        $buttonGrid = $this->createButtonGrid($formattedData['buttons'], 5);
        $this->telegram->sendMessageWithButtons($chatId, $formattedData['text'], $buttonGrid, $message['message_id']);
    }
    
    /**
     * Handle broadcast command
     */
    private function handleBroadcastCommand($chatId, $userId, $text) {
        if (!$this->security->isAdmin($userId)) {
            return;
        }
        
        $broadcastMessage = trim(substr($text, 10));
        $result = $this->broadcastManager->createBroadcast($userId, $broadcastMessage);
        
        $this->telegram->sendMessage($chatId, $result['message']);
    }
    
    /**
     * Handle broadcast confirm
     */
    private function handleBroadcastConfirm($chatId, $userId) {
        if (!$this->security->isAdmin($userId)) {
            return;
        }
        
        $result = $this->broadcastManager->executeBroadcast($this->getPendingBroadcastId($userId), $chatId);
        $this->telegram->sendMessage($chatId, $result['message']);
    }
    
    /**
     * Handle broadcast cancel
     */
    private function handleBroadcastCancel($chatId, $userId) {
        if (!$this->security->isAdmin($userId)) {
            return;
        }
        
        $result = $this->broadcastManager->cancelBroadcast($this->getPendingBroadcastId($userId), $userId);
        if ($result) {
            $this->telegram->sendMessage($chatId, "✅ <b>Broadcast dibatalkan!</b>");
        } else {
            $this->telegram->sendMessage($chatId, "❌ <b>Tidak ada broadcast yang bisa dibatalkan!</b>");
        }
    }
    
    /**
     * Handle stats command
     */
    private function handleStatsCommand($chatId, $userId) {
        if (!$this->security->isAdmin($userId)) {
            return;
        }
        
        try {
            $totalUsers = $this->userManager->getUserCount();
            $searchStats = $this->userManager->getSearchStats(30);
            
            $statsMessage = "📊 <b>STATISTIK BOT</b>\n\n";
            $statsMessage .= "👥 <b>Total User:</b> {$totalUsers}\n";
            
            foreach ($searchStats as $stat) {
                $type = $stat['search_type'] === 'movie' ? 'Film' : 'Adult';
                $statsMessage .= "🔍 <b>Pencarian {$type}:</b> {$stat['total_searches']}\n";
            }
            
            $statsMessage .= "\n⏰ <b>Update:</b> " . date('d/m/Y H:i:s') . " WIB";
            
            $this->telegram->sendMessage($chatId, $statsMessage);
            
        } catch (Exception $e) {
            $this->logger->error("Stats command failed", ['error' => $e->getMessage()]);
            $this->telegram->sendMessage($chatId, "❌ <b>Gagal mengambil statistik</b>");
        }
    }
    
    /**
     * Handle start/help command
     */
    private function handleStartHelpCommand($chatId, $userId, $text, $firstName, $username) {
        try {
            $totalFilms = $this->movieScraper->getTotalFilms();
            $totalAdult = $this->adultScraper->getTotalAdultContent();
            
            $userName = $firstName ?: $username ?: 'User';
            
            if ($text === '/start') {
                $message = "🎬 <b>Selamat Datang di Movie Club Film Bot!</b>\n\n";
                $message .= "🎉 <b>Halo {$userName}!</b>\n\n";
                $message .= "🎯 <b>Database:</b> <code>" . number_format($totalFilms) . "+</code> film\n";
                $message .= "🔞 <b>Adult:</b> <code>" . number_format($totalAdult) . "+</code> konten\n\n";
                $message .= "🔍 <b>Commands:</b>\n";
                $message .= "• <code>/cari [judul]</code> - Cari film\n";
                $message .= "• <code>/adult [query]</code> - Cari adult\n";
                $message .= "• <code>/adult random</code> - Random adult\n\n";
                $message .= "💡 <b>Klik /help untuk tutorial lengkap</b>";
            } else {
                $message = "📖 <b>TUTORIAL LENGKAP</b>\n\n";
                $message .= "🎯 <b>Database:</b> <code>" . number_format($totalFilms) . "+</code> film\n";
                $message .= "🔞 <b>Adult:</b> <code>" . number_format($totalAdult) . "+</code> konten\n\n";
                $message .= "🔍 <b>COMMANDS:</b>\n\n";
                $message .= "📝 <b>1. CARI FILM:</b>\n";
                $message .= "• <code>/cari avengers</code>\n";
                $message .= "• <code>/cari spiderman</code>\n\n";
                $message .= "🔞 <b>2. CARI ADULT:</b>\n";
                $message .= "• <code>/adult JAV</code>\n";
                $message .= "• <code>/adult random</code>\n\n";
                $message .= "🎮 <b>CARA PENGGUNAAN:</b>\n";
                $message .= "• Klik angka untuk detail\n";
                $message .= "• Pilih server streaming\n";
                $message .= "• Hanya 1 link valid, 2 lainnya iklan\n\n";
                $message .= "🚀 <b>Siap mulai?</b>";
            }
            
            $this->telegram->sendMessage($chatId, $message);
            
        } catch (Exception $e) {
            $this->logger->error("Start/help command failed", ['error' => $e->getMessage()]);
            $this->telegram->sendMessage($chatId, "❌ <b>Gagal memuat informasi</b>");
        }
    }
    
    /**
     * Handle text message
     */
    private function handleTextMessage($chatId) {
        $message = "🔍 <b>Untuk mencari konten, gunakan command:</b>\n\n";
        $message .= "<code>/cari [judul film]</code>\n";
        $message .= "<code>/adult [query]</code>\n\n";
        $message .= "💡 <b>Contoh:</b>\n";
        $message .= "<code>/cari avengers</code>\n";
        $message .= "<code>/adult JAV</code>";
        
        $this->telegram->sendMessage($chatId, $message);
    }
    
    /**
     * Handle callback queries
     */
    private function handleCallbackQuery($callbackQuery) {
        try {
            $chatId = $callbackQuery['message']['chat']['id'];
            $messageId = $callbackQuery['message']['message_id'];
            $data = $callbackQuery['data'];
            $userId = $callbackQuery['from']['id'];
            
            // Log to database
            $username = $callbackQuery['from']['username'] ?? 'unknown';
            $this->dbLogger->logBot('access', 'Callback query received', $userId, $username, 'callback', "data: {$data}");
            
            $this->logger->info("Callback query received", [
                'user_id' => $userId,
                'chat_id' => $chatId,
                'data' => $data
            ]);
            
            // Answer callback query
            $this->telegram->answerCallbackQuery($callbackQuery['id']);
            
            if (strpos($data, 'detail_') === 0) {
                $this->handleMovieDetailCallback($chatId, $data);
            } elseif (strpos($data, 'adult_') === 0) {
                $this->handleAdultDetailCallback($chatId, $data);
            } elseif ($data === 'adult_random_new') {
                $this->handleAdultRandomNewCallback($chatId, $messageId);
            } elseif ($data === 'broadcast_cancel_live') {
                $this->handleBroadcastCancelCallback($chatId, $messageId, $userId);
            }
            
        } catch (Exception $e) {
            $this->logger->error("Callback query handling failed", [
                'error' => $e->getMessage(),
                'data' => $data ?? 'unknown'
            ]);
        }
    }
    
    /**
     * Handle movie detail callback
     */
    private function handleMovieDetailCallback($chatId, $data) {
        $movieSlug = substr($data, 7);
        $url = 'https://tv6.lk21official.cc/' . $movieSlug;
        
        $this->telegram->sendMessage($chatId, "🔍 <b>Loading...</b>");
        
        $detail = $this->movieScraper->scrapeMovieDetail($url);
        
        if ($detail === false) {
            $this->telegram->sendMessage($chatId, "❌ <b>Gagal mengambil detail</b>\n\n💡 <b>Coba lagi:</b> <code>/cari [judul film]</code>");
            return;
        }
        
        $detailMessage = $this->formatMovieDetail($detail);
        
        // Create download buttons
        $buttons = $this->createDownloadButtons($movieSlug, 'movie');
        
        if (isset($detail['poster']) && !empty($detail['poster'])) {
            $this->telegram->sendPhoto($chatId, $detail['poster'], $detailMessage);
            if (!empty($buttons)) {
                $buttonGrid = $this->createButtonGrid($buttons, 1);
                $this->telegram->sendMessageWithButtons($chatId, "📥 <b>Link Streaming:</b>\n\n💡 <b>Klik tombol di bawah untuk streaming</b>", $buttonGrid);
            }
        } else {
            if (!empty($buttons)) {
                $buttonGrid = $this->createButtonGrid($buttons, 1);
                $this->telegram->sendMessageWithButtons($chatId, $detailMessage . "\n\n📥 <b>Link Streaming:</b>\n\n💡 <b>Klik tombol di bawah untuk streaming</b>", $buttonGrid);
            } else {
                $this->telegram->sendMessage($chatId, $detailMessage);
            }
        }
    }
    
    /**
     * Handle adult detail callback
     */
    private function handleAdultDetailCallback($chatId, $data) {
        $contentId = substr($data, 6);
        
        $this->telegram->sendMessage($chatId, "🔍 <b>Loading adult content...</b>");
        
        $content = $this->adultScraper->getAdultContentDetail($contentId);
        
        if (!$content) {
            $this->telegram->sendMessage($chatId, "❌ <b>Gagal mengambil detail konten adult</b>");
            return;
        }
        
        $detailMessage = $this->formatAdultDetail($content);
        
        // Create download buttons
        $buttons = $this->createDownloadButtons($content['embed_url'] ?? '', 'adult');
        
        if (isset($content['image_url']) && !empty($content['image_url'])) {
            $proxyImageUrl = 'https://i0.wp.com/' . str_replace(['http://', 'https://'], '', $content['image_url']);
            $this->telegram->sendPhoto($chatId, $proxyImageUrl, $detailMessage);
            if (!empty($buttons)) {
                $buttonGrid = $this->createButtonGrid($buttons, 1);
                $this->telegram->sendMessageWithButtons($chatId, "🔞 <b>Link Streaming:</b>\n\n💡 <b>Klik tombol di bawah untuk streaming</b>", $buttonGrid);
            }
        } else {
            if (!empty($buttons)) {
                $buttonGrid = $this->createButtonGrid($buttons, 1);
                $this->telegram->sendMessageWithButtons($chatId, $detailMessage . "\n\n🔞 <b>Link Streaming:</b>\n\n💡 <b>Klik tombol di bawah untuk streaming</b>", $buttonGrid);
            } else {
                $this->telegram->sendMessage($chatId, $detailMessage);
            }
        }
    }
    
    /**
     * Handle adult random new callback
     */
    private function handleAdultRandomNewCallback($chatId, $messageId) {
        $this->telegram->deleteMessage($chatId, $messageId);
        $this->telegram->sendMessage($chatId, "🔍 <b>Mengambil 10 adult content random...</b>");
        
        $randomAdultContent = $this->adultScraper->getRandomAdultContent();
        
        if (empty($randomAdultContent)) {
            $this->telegram->sendMessage($chatId, "❌ <b>Gagal mengambil adult content random</b>");
            return;
        }
        
        $messageText = "🔞 <b>Adult Random (10 Data)</b>\n\n";
        $messageText .= "📊 <b>Total:</b> " . count($randomAdultContent) . " data\n\n";
        
        $buttons = [];
        foreach ($randomAdultContent as $index => $content) {
            $buttonNumber = $index + 1;
            $messageText .= "{$buttonNumber}. <b>" . htmlspecialchars($content['title']) . "</b>\n";
            $messageText .= "   📅 {$content['Tahun']} | 🎬 {$content['Kualitas']} | ⏱️ {$content['Durasi']}\n\n";
            
            $buttons[] = [
                'text' => $buttonNumber,
                'callback_data' => 'adult_' . $content['id']
            ];
        }
        
        $buttons[] = [
            'text' => '🎲 Random Lagi',
            'callback_data' => 'adult_random_new'
        ];
        
        $buttonGrid = $this->createButtonGrid($buttons, 5);
        $this->telegram->sendMessageWithButtons($chatId, $messageText, $buttonGrid);
    }
    
    /**
     * Handle broadcast cancel callback
     */
    private function handleBroadcastCancelCallback($chatId, $messageId, $userId) {
        // Create cancel flag
        file_put_contents('broadcast_cancelled.flag', time());
        
        $cancelMessage = "❌ <b>Broadcast Dibatalkan!</b>\n\n";
        $cancelMessage .= "⏹️ <b>Proses broadcast sedang dihentikan...</b>\n";
        $cancelMessage .= "📊 <b>Status:</b> Dibatalkan oleh admin\n";
        $cancelMessage .= "⏰ <b>Waktu:</b> " . date('H:i:s') . " WIB\n\n";
        $cancelMessage .= "🔄 <b>Menunggu proses selesai...</b>";
        
        $this->telegram->editMessage($chatId, $messageId, $cancelMessage);
        
        $this->logger->info("Broadcast cancelled by admin", ['admin_id' => $userId]);
    }
    
    /**
     * Format movie message with buttons
     */
    private function formatMovieMessageWithButtons($movies, $query) {
        if (empty($movies)) {
            return [
                'text' => "❌ <b>Tidak ada film ditemukan</b>\n\n💡 <b>Coba:</b> <code>/cari avengers</code>",
                'buttons' => []
            ];
        }
        
        $message = "🎬 <b>Hasil: {$query}</b>\n\n";
        $buttons = [];
        
        foreach ($movies as $index => $movie) {
            $num = $index + 1;
            $title = htmlspecialchars($movie['title']);
            $year = isset($movie['year']) ? " ({$movie['year']})" : "";
            $rating = isset($movie['rating']) ? " ⭐{$movie['rating']}" : "";
            $quality = isset($movie['quality']) ? " [{$movie['quality']}]" : "";
            
            $message .= "{$num}. <b>{$title}</b>{$rating}{$quality}\n";
            
            if (isset($movie['url'])) {
                $movieId = $movie['slug'] ?? $index;
                $callbackData = "detail_" . $movieId;
                if (strlen($callbackData) > 64) {
                    $maxSlugLength = 64 - 7;
                    $truncatedSlug = substr($movieId, 0, $maxSlugLength);
                    $callbackData = "detail_" . $truncatedSlug;
                }
                $buttons[] = [
                    'text' => "{$num}",
                    'callback_data' => $callbackData
                ];
            }
        }
        
        return [
            'text' => $message,
            'buttons' => $this->createButtonGrid($buttons, 5)
        ];
    }
    
    /**
     * Format adult content message with buttons
     */
    private function formatAdultContentMessageWithButtons($adultContent, $query) {
        if (empty($adultContent)) {
            return [
                'text' => "❌ <b>Tidak ada konten ditemukan</b>\n\n💡 <b>Coba:</b> <code>/adult JAV</code>",
                'buttons' => []
            ];
        }
        
        $message = "🔞 <b>Hasil Adult: {$query}</b>\n\n";
        $buttons = [];
        
        foreach ($adultContent as $index => $content) {
            $num = $index + 1;
            $title = htmlspecialchars($content['title']);
            $year = isset($content['year']) ? " ({$content['year']})" : "";
            $quality = isset($content['quality']) ? " [{$content['quality']}]" : "";
            $genre = isset($content['genre']) ? " - {$content['genre']}" : "";
            
            $message .= "{$num}. <b>{$title}</b>{$year}{$quality}{$genre}\n";
            
            if (isset($content['embed_url']) && !empty($content['embed_url'])) {
                $contentId = $content['id'] ?? $index;
                $callbackData = "adult_" . $contentId;
                if (strlen($callbackData) > 64) {
                    $maxIdLength = 64 - 6;
                    $truncatedId = substr($contentId, 0, $maxIdLength);
                    $callbackData = "adult_" . $truncatedId;
                }
                $buttons[] = [
                    'text' => "{$num}",
                    'callback_data' => $callbackData
                ];
            }
        }
        
        return [
            'text' => $message,
            'buttons' => $this->createButtonGrid($buttons, 5)
        ];
    }
    
    /**
     * Format movie detail
     */
    private function formatMovieDetail($detail) {
        if (!$detail) {
            return "❌ Gagal mengambil detail film.";
        }
        
        $message = "🎬 <b>" . htmlspecialchars($detail['title']) . "</b>\n";
        
        if (isset($detail['rating'])) {
            $message .= "⭐ {$detail['rating']}";
        }
        
        if (isset($detail['info_tags']) && !empty($detail['info_tags'])) {
            $message .= " • " . implode(' • ', $detail['info_tags']);
        }
        
        $message .= "\n";
        
        if (isset($detail['genres']) && !empty($detail['genres'])) {
            $message .= "🎭 " . implode(', ', $detail['genres']) . "\n";
        }
        
        if (isset($detail['synopsis']) && !empty($detail['synopsis'])) {
            $synopsis = htmlspecialchars($detail['synopsis']);
            if (strlen($synopsis) > 200) {
                $synopsis = substr($synopsis, 0, 200) . "...";
            }
            $message .= "\n📝 {$synopsis}";
        }
        
        return $message;
    }
    
    /**
     * Format adult detail
     */
    private function formatAdultDetail($content) {
        $message = "🔞 <b>" . htmlspecialchars($content['title']) . "</b>\n\n";
        
        if (isset($content['year']) && !empty($content['year'])) {
            $message .= "📅 <b>Tahun:</b> {$content['year']}\n";
        }
        
        if (isset($content['quality']) && !empty($content['quality'])) {
            $message .= "🎬 <b>Kualitas:</b> {$content['quality']}\n";
        }
        
        if (isset($content['duration']) && !empty($content['duration'])) {
            $message .= "⏱️ <b>Durasi:</b> {$content['duration']}\n";
        }
        
        if (isset($content['genre']) && !empty($content['genre'])) {
            $message .= "🎭 <b>Genre:</b> {$content['genre']}\n";
        }
        
        return $message;
    }
    
    /**
     * Create download buttons
     */
    private function createDownloadButtons($url, $type = 'movie') {
        if (empty($url)) {
            return [];
        }
        
        $encodedUrl = base64_encode($url);
        $encodedAds3 = base64_encode('ads3');
        
        $downloadUrl = "{$this->publicDomain}/dl.php?url={$encodedUrl}&ref={$encodedAds3}";
        
        $icon = $type === 'movie' ? '📥' : '🎬';
        
        return [
            [
                'text' => "{$icon} Streaming",
                'url' => $downloadUrl
            ]
        ];
    }
    
    /**
     * Create button grid
     */
    private function createButtonGrid($buttons, $buttonsPerRow = 5) {
        if (empty($buttons)) {
            return [];
        }
        
        if (isset($buttons[0]) && isset($buttons[0]['text'])) {
            return [array_chunk($buttons, $buttonsPerRow)[0]];
        }
        
        return $buttons;
    }
    
    /**
     * Get update type
     */
    private function getUpdateType($update) {
        if (isset($update['message'])) return 'message';
        if (isset($update['callback_query'])) return 'callback_query';
        if (isset($update['inline_query'])) return 'inline_query';
        if (isset($update['chosen_inline_result'])) return 'chosen_inline_result';
        return 'unknown';
    }
    
    /**
     * Get pending broadcast ID for user
     */
    private function getPendingBroadcastId($userId) {
        try {
            $pdo = $this->db->getConnection();
            $stmt = $pdo->prepare("SELECT id FROM broadcast_logs WHERE admin_id = ? AND status = 'pending' ORDER BY created_at DESC LIMIT 1");
            $stmt->execute([$userId]);
            $result = $stmt->fetch();
            return $result ? $result['id'] : null;
        } catch (Exception $e) {
            $this->logger->error("Failed to get pending broadcast ID", ['error' => $e->getMessage()]);
            return null;
        }
    }
}

// Initialize and run bot
try {
    $botHandler = new BotHandler(
        $db, $logger, $security, $telegram, 
        $userManager, $broadcastManager, 
        $movieScraper, $adultScraper, $publicDomain
    );
    
    $botHandler->processWebhook();
    
} catch (Exception $e) {
    $logger->error("Bot execution failed", [
        'error' => $e->getMessage(),
        'trace' => $e->getTraceAsString()
    ]);
    http_response_code(500);
}
